(ns genesis.providers.aws.elb.listener
  (:require [amazonica.aws.elasticloadbalancingv2 :as elb]
            [genesis.providers.aws.ec2 :as gec2]
            [genesis.providers.aws.elb.load-balancer :as load-balancer]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]
            [clojure.data :as data]
            [genesis.core :as g :refer [defresource]]
            [genesis.util :refer [validate!]]))

(s/def :action/type #{"forward"})
(s/def ::target-group-arn string?)

(s/def ::action (s/keys :req-un [:action/type
                                 ::target-group-arn]))

(s/def ::certificate (s/keys :req-un [::certificate-arn]))
(s/def ::certificates (s/coll-of ::certificate))

(s/def ::listener (s/keys :req-un [;; ::instance-port
                                   ::protocol
                                   ::default-actions
                                   ::load-balancer-arn
                                   ::port]
                          :opt-un [::certificates]))

(s/def ::listeners (s/coll-of ::listener))

(def list-listeners- (gec2/list-with-identity elb/describe-listeners {:id-key :listener-arn
                                                                      :require-args? true}))

(defn list-listeners [context]
  (let [creds (-> context :aws :creds)]
    (some->> (load-balancer/list-load-balancers context)
             (mapcat (fn [lb]
                       (assert (-> lb :properties :load-balancer-arn))
                       (-> (elb/describe-listeners creds {:load-balancer-arn (-> lb :properties :load-balancer-arn)})
                           :listeners)))
             (seq)
             (map (fn [l]
                    {:resource :elb/listener
                     :identity (:listener-arn l)
                     :properties l})))))

(def get-listener (gec2/get-by-identity list-listeners))

(def create-listener (gec2/create-with-identity elb/create-listener {:spec ::listener
                                                                     :id-key :listener-arn
                                                                     :unwrap-array? true}))

(def delete-listener (gec2/delete-by-identity {:get-fn get-listener
                                               :delete-fn elb/delete-listener
                                               :id-key :listener-arn}))

(defresource :elb/listener {:create create-listener
                            :list list-listeners
                            :get get-listener
                            :delete delete-listener})
