(ns genesis.state.s3
  (:require [cognitect.aws.client.api :as aws]
            [cognitect.aws.credentials :as credentials]
            [clojure.spec.alpha :as s]
            [clojure.edn :as edn]
            [genesis.providers.aws :as gaws]
            [genesis.specs :as gs]
            [genesis.state :as state]
            [genesis.state.fs :as fs]
            [genesis.util :refer [instrument-ns ref-of]])
  (:import java.security.MessageDigest
           java.math.BigInteger))

(s/def ::index-name (s/map-of ::gs/key ::gs/existing-instance))
(s/def ::index-identity (s/map-of ::gs/identity ::gs/existing-instance))
(s/def ::disk-db (s/keys :req-un [::index-identity]))
(s/def ::db (ref-of (s/keys :req-un [::index-name ::index-identity])))
(s/def ::state (s/keys :req-un [::db ::key ::bucket]))

(defn md5 [s]
  {:body [(do (println "md5" %) true)]}
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s "UTF-8"))]
    (format "%032x" (BigInteger. 1 raw))))

(s/def ::bucket string?)
(s/def ::key string?)

(s/fdef save! :args (s/cat :state ::state))
(defn save! [{:keys [key bucket client db]}]
  (let [body (-> @db :index-identity vals vec pr-str)]
    (-> client
        (gaws/invoke! {:op :PutObject :request {:Bucket bucket
                                                :Key key
                                                :Body (-> body
                                                          (.getBytes "UTF-8")
                                                          (java.io.ByteArrayInputStream.))
                                                ;; :ContentMD5 (md5 body)
                                                }}))))

(defn assert-success! [response]
  (if (-> response :cognitect.anomalies/category)
    (throw (ex-info "aws api request failed" response))
    response))

(defn load! [{:keys [key bucket client]}]
  (let [instances (-> client
                      (aws/invoke {:op :GetObject :request {:Bucket bucket
                                                            :Key key}})
                      (assert-success!)
                      :Body
                      slurp
                      edn/read-string
                      (->>
                       (map (fn [i]
                              [(:identity i) i]))
                       (into {})))]
    (->
     (ref {:index-identity instances})
     (fs/index!))))

(defrecord S3State [bucket key client db]
  state/State
  (init- [this])
  (list- [this]
    (->> (get-in @db [:index-identity])
         vals))
  (create- [this instance]
    (dosync
     (alter db assoc-in [:index-identity (:identity instance)] instance)
     (fs/index! db))
    (save! {:bucket bucket :key key :client client :db db}))
  (get- [this ident]
    (get-in @db [:index-identity ident]))
  (get-by-name- [this resource name]
    (get-in @db [:index-name [resource name]]))
  (delete- [this ident]
    (dosync
     (alter db update-in [:index-identity] dissoc ident))
    (fs/index! db)
    (save! {:bucket bucket :key key :client client :db db})))

(s/fdef new :args (s/cat :a (s/keys :req-un [::bucket ::key] :opt-un [:aws/creds])))
(defn new [{:keys [bucket key region creds]}]
  (let [client (aws/client (merge
                            {:api :s3}
                            (when region
                              {:region region})
                            (when creds
                              {:credentials-provider (credentials/basic-credentials-provider
                                                      creds)})))]
    (aws/validate-requests client true)
    (map->S3State {:bucket bucket
                   :key key
                   :client client
                   :db (load! {:bucket bucket :key key :client client})})))

(instrument-ns)
