(ns gv.view
  (:require [seesaw.core :refer :all]
            [clojure.java.shell :refer [sh]]
            [clojure.java.io :as io])
  (:import [javax.swing ImageIcon]
           [javax.imageio ImageIO]))

(defn- new-imageion [bytes]
  (new ImageIcon bytes))

(defn- bytes->imageicon [bytes]
  (new-imageion bytes))

(defn- icon-data [^ImageIcon icon]
  {:height (.getIconHeight icon)
   :width (.getIconWidth icon)})

(defn- scale-icon
  ([^ImageIcon icon scale]
   (let [{:keys [height width]} (icon-data icon)
         height (* height scale)
         width (* width scale)]
     (scale-icon icon width height)))
  ([^ImageIcon icon x y]
   (let [image (.getImage icon)
         image (.getScaledInstance image x y java.awt.Image/SCALE_SMOOTH)]
     (new ImageIcon image))))

(defn- visualize-icon! [^ImageIcon icon w h]
  (frame
   :visible? true
   :width w
   :height h
   :content (canvas :paint (fn [c g] (.paintIcon icon c g 0 0)))))

;; dot

(defn- dot-string->bytes
  [s &{:keys [format] :or {format "png"}}]
  (let [{:keys [exit out err]} (sh "dot" (str "-T" format)
                                   :in s
                                   :out-enc :bytes)]
    (if (= exit 0)
      out
      (throw (ex-info "dot compile error"
                      {:message err})))))

(defn view-dot!
  "s: string of a valid dot program.
   scale: scaling parameter.

  Result: displays an image in a window."
  [s scale]
  (let [bytes (dot-string->bytes s)
        icon (bytes->imageicon bytes)
        icon (scale-icon icon scale)
        {:keys [width height]} (icon-data icon)
        [w h] (map #(+ % 50) [width height])]
    (visualize-icon! icon w h)))

;; create and save files

(defn- bytes->image
  "Returns a BufferedImage."
  [bytes]
  (ImageIO/read (io/input-stream bytes)))

(defn- save-image [image filename]
  (ImageIO/write image "png" (io/file filename)))

(defn save-dot!
  "Takes a string containing a valid dot program and a filename.
  Ssaves the resulting image in the file."
  [s filename]
  (let [bytes (dot-string->bytes s)
        image (bytes->image bytes)]
    (save-image image filename)))
