(ns gv.core
  (:require 
   [clojure.spec.alpha :as s]
   [clojure.zip :as z]
   [medley.core :refer [filter-vals]]
   [better-cond.core :refer [defnc defnc- cond] :rename {cond bcond}]
   [gv.view :refer [view-dot! save-dot!]]
   [gv.data :refer :all]))

;;TODO: Be kind to the user and improve error message when
;; a false graph is created. 

(def ^{:private true} nl "\n")

(def ^{:private true} linesep (str ";" nl))

(def ^{:private true} comma ",")

(defn- wrap [s] (format "\"%s\"" s))

(defn- text [x]
  (cond (string? x) (wrap x)
        (keyword? x) (name x)
        :else (str x)))

(defn- branket [& args] (format "[%s]" (reduce str args)))

(defn- curly [& args] ( format "{%s}" (reduce str args)))

(defprotocol Dot
  (dot [g] "converts a graph to a dot string." ))

(defn view
  "displays a graph in a wondow."
  [g &{:keys [scale] :or {scale 1}}] (view-dot! (dot g) (* 2 scale)))

(defn- attr->str [m]
  (branket
   (reduce
    str
    (interpose ","
               (map (fn [[k v]]
                      (format "%s = %s" (name k) (text v)))
                    m)))))

(defn- attr->str-sans-branket [m]
  (reduce str (for [[k v] m] (str (name k) " = " (text v) ";" nl))))

(defn- node->str [x]
  (if (coll? x) (let [[n m] x] (str n " "(attr->str m))) (text x)))

(defn- edge->str [[from to m] directed?]
  (let [arrow (if directed? " -> "  " -- ")]
    (str (node->str from) arrow (node->str to)
         (when m (str " " (attr->str m))))))

(defn- nodes->str [coll]
  (reduce str (for [n coll] (str (node->str n) ";" nl))))

(defn- edges->str [coll directed?]
  (reduce str (for [e coll] (str (edge->str e directed?) ";" nl))))

(def ^{:private true}
  main-keys
  #{:nodes :edges :node-attr :edge-attr :graph-attr :clusters})

(defn- graph-attrs->str [attrs]
  (reduce str (map (fn [[k v]] (str (name k) " = " (text v) ";" nl)) attrs)))

(defn- cluster-name [] (format "subgraph cluster_%s" (gensym)))

(defn- handle-attrs [m]
  (let [{:keys [node-attr edge-attr graph-attr attrs]} m
        sep (str ";" nl)]
    (str
     (when (not-empty attrs) (attr->str-sans-branket attrs))
     (when (not-empty node-attr) (str "node " (attr->str node-attr) sep))
     (when (not-empty edge-attr) (str "edge " (attr->str edge-attr) sep))
     (when (not-empty graph-attr) (str "graph " (attr->str graph-attr) sep)))))

(defn- graph->str [m head directed?]
  (let [{:keys [nodes edges clusters]} m]
    (str head
         nl
         (curly
          (handle-attrs m)
          (nodes->str nodes)
          (edges->str edges directed?)
          (reduce str
                  (for [g clusters] (str (dot g) ";" nl)))))))

;; define specs

(s/def ::node-attr (s/map-of (set node-attributes) any?))

(s/def ::edge-attr (s/map-of (set edge-attributes) any?))

(s/def ::graph-attr (s/map-of (set graph-attributes) any?))

(s/def ::node (s/or
               :s string?
               :i number?
               :k keyword?
               :with-attr (s/cat :n ::node :m ::node-attr)))

(defn- node->str [x]
  (case (key (s/conform ::node x))
    (:s :i :k) (text x)
    :with-attr (let [[n m] x] (str (node->str n) " " (attr->str m)))))

(s/def ::edge (s/cat :from ::node :to ::node :m (s/?  ::edge-attr )))

(defn- edge->str [x directed?]
  (let [edge-sep (if directed? " -> " " -- ")
        [from to m] x]
    (str (node->str from) edge-sep (node->str to) (when m (str " " (attr->str m))))))

(s/def ::nodes (s/coll-of ::node))

(s/def ::edges (s/coll-of ::edge))

(s/def ::clusters (s/coll-of ::graph))

(s/def ::graph
  (s/keys :opt-un
          [::nodes ::edges ::node-attr ::edge-attr ::graph-attr ::clusters]))

(defn- valid-graph? [x] (s/valid? ::graph x))

(def ^{:private true} main-keys
  #{:nodes :edges :node-attr :edge-attr :graph-attr :clusters
    :directed? :type})

(defn- nodes->str [coll]
  (reduce str (for [n coll] (str (node->str n) ";" nl))))

(defn- edges->str [coll directed?]
  (reduce str
          (for [e coll] (str (edge->str e directed?) ";" nl))))

(defnc- global-attrs->str [g]
  let [{:keys [node-attr edge-attr graph-attr]} g
       others (reduce dissoc g main-keys)
       sep (str ";" nl)
       f (fn [m]
           (reduce str (for [[k v] m] (format "%s = %s;\n"
                                              (text k) (text v)))))]
  (str
   (when (not-empty node-attr)
     (str "node " (attr->str node-attr) sep))

   (when (not-empty edge-attr)
     (str "edge " (attr->str edge-attr) sep))

   (when (not-empty graph-attr)
     (str "graph " (attr->str graph-attr) sep))

   (f others)))

(defn- handle-clusters
  "assocs a :directed? key-val to all the subgraphs and `g`.
  A subgraph is directed iff its parent is so."
  [g]
  (let [directed? (= (:type g) :directed)
        z (z/zipper :clusters :clusters #(assoc %1 :clusters %2) g)
        z (z/edit z assoc :directed? directed?)]
    (loop [loc z]
      (let [loc (if (-> loc z/node :type (= :cluster))
                  (z/edit loc assoc :directed? directed?)
                  loc)]
        (if (z/end? loc) (z/root loc) (recur (z/next loc)))))))

(defn- cluster-name []
  (format "subgraph cluster_%s" (gensym)))

(defn- graph->str [g]
  (letfn [(go [g*]
              (bcond
               let [{:keys [directed? edges nodes]} g*
                    cluster? (-> g* :type (= :cluster))
                    title (cond
                            cluster? (cluster-name)
                            directed? "digraph"
                            :else "graph")
                    clusters-s (reduce str
                                       (for [g** (:clusters g*)]
                                         (str (go g**) ";" nl)))]
               (str title
                    (curly
                     (global-attrs->str g*)
                     (nodes->str nodes)
                     (edges->str edges directed?)
                     clusters-s))))]
    (-> g handle-clusters go)))

;; ctor

(defn- check-validity
  "throws if `g` is not a valid graph. or returns g."
  [g]
  (if (valid-graph? g)
    g
    (throw (Exception.
            (str g " is not a valid graph."
                 nl nl
                 "explanation:"
                 nl nl
                 (s/explain-str ::graph g))))))

(defn- new-graph [type m]
  (check-validity (assoc m :type type)))

(defn digraph
  "Creates a new directed graph.
  See also gv.core/graph ."
  [& args]
  (new-graph :directed (apply hash-map args)))

(defn graph
  "Creates a graph.
  Example:

  (graph
   :nodes [1 2 3 4 5]
   :edges [[1 2] [2 3 {:color :red}]]
   :clusters [(cluster
               :label \"This is a cluster.\"
               :nodes [4 5]
               :edges[[4 5]])]))"
  [& args]
  (new-graph :graph (apply hash-map args)))

(defn cluster
  "Creates a new cluster.

  See also gv.core/graph ."
  [& args]
  (new-graph :cluster (apply hash-map args)))

;; dot and view

(defn dot
  "Given a graph, returns the dot string of it."
  [g] (graph->str g))

(defn view
  "Displays a graph in a window."
  [g &{:keys [scale] :or {scale 1}}]
  (view-dot! (dot g) (* scale 2)))

(defn attributes-data []
  {:all attributes
   :node-attributes node-attributes
   :edge-attributes edge-attributes
   :graph-attributes graph-attributes})

(comment

  (let [g (digraph
           :label "test"
           :nodes [1 2 3]
           :clusters [(cluster
                       :label "sub"
                       :nodes [1 2]
                       :edges [[1 2 {:color "red"}]])])]
    (view g))

  )


