(ns hara.core.base.thread)

(defn current-thread
  "returns the current thread"
  {:added "3.0"}
  ([]
   (Thread/currentThread)))

(defn interrupt
  "interrupts a thread"
  {:added "3.0"}
  ([^Thread thread]
   (.interrupt thread)))
  
(defn sleep
  "sleeps for n milliseconds"
  {:added "3.0"}
  ([^long ms]
   (Thread/sleep ms)))

(defn wait-spin
  "waits using onSpin"
  {:added "3.0"}
  ([]
   (Thread/onSpinWait)))

(defn wait-on
  "waits for a lock to notify"
  {:added "3.0"}
  ([^Object lock]
   (locking lock (.wait lock))))

(defn notify
  "notifies threads waiting on lock"
  {:added "3.0"}
  ([^Object lock]
   (locking lock (.notify lock))))

(defn notify-all
  "notifies all threads waiting on lock"
  {:added "3.0"}
  ([^Object lock]
   (locking lock (.notifyAll lock))))

(defn has-lock?
  "checks if thread has the lock"
  {:added "3.0"}
  ([^Object lock]
   (Thread/holdsLock lock)))

(defn yield
  "calls yield on current thread"
  {:added "3.0"}
  ([]
   (Thread/yield)))

(defn stacktrace
  "returns thread stacktrace"
  {:added "3.0"}
  ([]
   (stacktrace (current-thread)))
  ([^Thread thread]
   (.getStackTrace thread)))

(defn all-stacktraces
  "returns all available stacktraces"
  {:added "3.0"}
  ([]
   (Thread/getAllStackTraces)))

(defn all-threads
  ([]
   (keys (all-stacktraces))))

(defn all-thread-ids
  ([]
   (set (map #(.getId ^Thread %) (all-threads)))))

(defn dump
  "dumps out current thread information"
  {:added "3.0"}
  ([]
   (Thread/dumpStack)))

(defn active-count
  "returns active threads"
  {:added "3.0"}
  ([]
   (Thread/activeCount)))

(defn alive?
  "checks if thread is alive"
  {:added "3.0"}
  ([^Thread thread]
   (.isAlive thread)))

(defn thread-id
  ([] (thread-id (current-thread)))
  ([^Thread thread]
   (.getId thread)))

(defn daemon?
  "checks if thread is a daemon"
  {:added "3.0"}
  ([]
   (daemon? (current-thread)))
  ([^Thread thread]
   (.isDaemon thread)))

(defn interrupted?
  "checks if thread has been interrupted"
  {:added "3.0"}
  ([^Thread thread]
   (.isInterrupted thread)))

(defn has-access?
  "checks if thread allows access to current"
  {:added "3.0"}
  ([^Thread thread]
   (try
     (.checkAccess thread)
     true
     (catch SecurityException e
       false))))

(defn start
  "starts a thread"
  {:added "3.0"}
  ([^Thread thread]
   (.start thread)))

(defn run
  "runs the thread function locally"
  {:added "3.0"}
  ([^Thread thread]
   (.run thread)))

(defn join
  "calls join on a thread"
  {:added "3.0"}
  ([]
   (join (current-thread)))
  ([^Thread thread]
   (.join thread))
  ([^Thread thread millis]
   (.join thread millis))
  ([^Thread thread millis nanos]
   (.join thread millis nanos)))

(defn exception-handler
  "returns the uncaught exception handler"
  {:added "3.0"}
  ([]
   (exception-handler (current-thread)))
  ([^Thread thread]
   (.getUncaughtExceptionHandler thread)))

(defn set-exception-handler!
  "sets the uncaught exception handler"
  {:added "3.0"}
  ([^Thread thread f]
   (.setUncaughtExceptionHandler thread f)))

(defn global-exception-handler
  "returns the global uncaught exception handler"
  {:added "3.0"}
  ([]
   (Thread/getDefaultUncaughtExceptionHandler)))

(defn set-global-exception-handler!
  "sets the global uncaught exception handler"
  {:added "3.0"}
  ([f]
   (Thread/setDefaultUncaughtExceptionHandler f)))

(defn classloader
  "returns the context classloader"
  {:added "3.0"}
  ([]
   (classloader (current-thread)))
  ([^Thread thread]
   (.getContextClassLoader thread)))

(defn set-classloader
  "sets the context classloader"
  {:added "3.0"}
  ([loader]
   (set-classloader (current-thread)))
  ([^Thread thread loader]
   (.setContextClassLoader thread loader)))

(defn ^Thread thread
  "creates a new thread"
  {:added "3.0"}
  ([{:keys [^Runnable handler ^bool daemon priority classloader exception-handler name start]}]
   (cond-> (Thread. handler)
     name        (doto (.setName name))
     daemon      (doto (.setDaemon daemon))
     priority    (doto (.setPriority priority))
     classloader (doto (.setContextClassLoader classloader))
     exception-handler (doto (.setUncaughtExceptionHandler exception-handler))
     start (doto (.start)))))
