(ns hara.core.component.common
  (:require [hara.util :as u]
            [hara.protocol.component :as protocol.component]))

(def ^:dynamic *kill* false)

(defn system?
  "checks if a component extends ISystem
 
   (system? (Database.))
   => false"
  {:added "3.0"}
  [obj]
  (satisfies? protocol.component/ISystem obj))

(defn component?
  "checks if an instance extends IComponent
 
   (component? (Database.))
   => true"
  {:added "3.0"}
  [obj]
  (u/suppress (extends? protocol.component/IComponent (type obj))))

(extend-protocol protocol.component/IComponent
  Object
  (-start [this] this)
  (-stop  [this] this)
  (-kill  [this] this)
  (-info  [this level] {})
  (-remote? [this] false)
  (-health  [this] {:status :ok}))
    
(defn primitive?
  "checks if a component is a primitive type
 
   (primitive? 1) => true
 
   (primitive? {}) => false"
  {:added "3.0"}
  [x]
  (or (string? x)
      (number? x)
      (boolean? x)
      (u/regexp? x)
      (uuid? x)
      (uri? x)
      (u/url? x)))

(defn started?
  "checks if a component has been started
 
   (started? 1)
   => true
 
   (started? {})
   => false"
  {:added "3.0"}
  [component]
  (try (protocol.component/-started? component)
       (catch IllegalArgumentException e
         (if (u/iobj? component)
           (-> component meta :component/started true?)
           (primitive? component)))
       (catch AbstractMethodError e
         (if (u/iobj? component)
           (-> component meta :component/started true?)
           (primitive? component)))))

(defn stopped?
  "checks if a component has been stopped
 
   (stopped? 1)
   => false
 
   (stopped? {})
   => true"
  {:added "3.0"}
  [component]
  (try (protocol.component/-stopped? component)
       (catch IllegalArgumentException e
         (-> component started? not))
       (catch AbstractMethodError e
         (-> component started? not))))

(defn perform-hooks
  "perform hooks before main function
 
   (perform-hooks (Database.)
                  {:init (fn [x] 1)}
                  [:init])
   => 1"
  {:added "3.0"}
  [component functions hook-ks]
  (reduce (fn [out k]
            (let [func (or (get functions k)
                           identity)]
              (func out)))
          component
          hook-ks))

(defn get-options
  "helper function for start and stop
 
   (get-options (Database.)
                {:init (fn [x] 1)})
   => (contains {:init fn?})"
  {:added "3.0"}
  [component opts]
  (let [mopts (cond (system? component)
                    (-> (meta component)
                        (select-keys [:setup :hooks :functions :teardown]))

                    (u/iobj? component)
                    (-> (meta component)
                        (u/qualified-keys :component)
                        (u/unqualify-keys)))]
    (merge mopts opts)))

(defn stop-raw
  "switch between stop and kill methods"
  {:added "3.0"}
  [component]
  (if *kill*
    (try (protocol.component/-kill component)
         (catch IllegalArgumentException e
           (protocol.component/-stop component))
         (catch AbstractMethodError e
           (protocol.component/-stop component)))
    (protocol.component/-stop component)))

(defn start
  "starts a component/array/system
 
   (start (Database.))
   => {:status \"started\"}"
  {:added "3.0"}
  ([component]
   (start component {}))
  ([component opts]
   (let [{:keys [setup hooks functions]} (get-options component opts)
         {:keys [pre-start post-start]} hooks
         functions (or (get component :functions) functions)
         setup     (or setup identity)
         component   (-> component
                         (perform-hooks functions pre-start)
                         (protocol.component/-start)
                         (setup)
                         (perform-hooks functions post-start))]
     (if (u/iobj? component)
       (vary-meta component assoc :component/started true)
       component))))

(defn stop
  "stops a component/array/system
 
   (stop (start (Database.))) => {}"
  {:added "3.0"}
  ([component]
   (stop component {}))
  ([component opts]
   (let [{:keys [teardown hooks functions]} (get-options component opts)
         {:keys [pre-stop post-stop]} hooks
         functions (or (get component :functions) functions)
         teardown  (or teardown identity)
         component (-> component
                       (perform-hooks functions pre-stop)
                       (teardown)
                       (stop-raw)
                       (perform-hooks functions post-stop))]
     (if (u/iobj? component)
       (vary-meta component assoc :component/started false)
       component))))

(defn kill
  "kills a systems, or if method is not defined stops it
 
   (kill (start (Database.))) => {}"
  {:added "3.0"}
  ([component]
   (kill component {}))
  ([component opts]
   (binding [*kill* true]
     (stop component opts))))

(defn info
  "returns info regarding the component
 
   (info (Database.))
   => {:info true}"
  {:added "3.0"}
  ([component]
   (info component :default))
  ([component level]
   (try (protocol.component/-info component level)
        (catch IllegalArgumentException e
          {})
        (catch AbstractMethodError e
          {}))))

(defn health
  "returns the health of the component
 
   (health (Database.))
   => {:status :ok}"
  {:added "3.0"}
  [component]
  (try (protocol.component/-health component)
       (catch AbstractMethodError e
         (throw e))
       (catch Throwable t
         {:status :errored})))

(defn remote?
  "returns whether the component connects remotely
 
   (remote? (Database.))
   => false"
  {:added "3.0"}
  ([component]
   (try (protocol.component/-remote? component)
        (catch AbstractMethodError e
          false))))   

(defn all-props
  "lists all props in the component
 
   (all-props (Database.))
   => [:interval]"
  {:added "3.0"}
  ([component]
   (try (keys (protocol.component/-props component))
        (catch AbstractMethodError e))))

(defn get-prop
  "gets a prop in the component
 
   (get-prop (Database.) :interval)
   => 10"
  {:added "3.0"}
  ([component k]
   (let [getter (-> (protocol.component/-props component)
                    (get-in [k :get]))]
     (getter))))

(defn set-prop
  "sets a prop in the component
 
   (set-prop (Database.) :interval 3)
   => throws"
  {:added "3.0"}
  ([component k value]
   (let [setter (-> (protocol.component/-props component)
                    (get-in [k :set]))]
     (setter value))))

(defmacro with
  "do tests with an active component
 
   (with [db (Database.)]
     (started? db))
   => true"
  {:added "3.0"}
  ([[var expr & more] & body]
   `(let [~var  ~expr
          ~var  (start ~var)]
      (try
        ~(if (empty? more)
           `(do ~@body)
           `(with [~@more] ~@body))
        (finally (stop ~var))))))
