(ns hara.data.base.seq
  (:require [hara.util :as u]))

(defn positions
  "find positions of elements matching the predicate
 
   (positions even? [5 5 4 4 3 3 2 2])
   => [2 3 6 7]"
  {:added "3.0"}
  ([pred coll]
   (keep-indexed (fn [idx x]
                   (when (pred x)
                     idx))
                 coll)))

(defn insert-index
  "insert one or more elements at the given index
 
   (insert-index [:a :b] 1 :b :c)
   => [:a :b :c :b]"
  {:added "3.0"}
  ([coll i new]
   (apply conj (subvec coll 0 i) new (subvec coll i)))
  ([coll i new & more]
   (apply conj (subvec coll 0 i) new (concat more (subvec coll i)))))

(defn remove-index
  "removes element at the specified index
 
   (remove-index [:a :b :c :d] 2)
   => [:a :b :d]"
  {:added "3.0"}
  ([coll i]
   (remove-index coll i 1))
  ([coll i n]
   (cond (vector? coll)
         (reduce conj
                 (subvec coll 0 i)
                 (subvec coll (+ i n) (count coll)))

         :else
         (keep-indexed #(if (not (and (>= %1 i)
                                      (< %1 (+ i n)))) %2) coll))))

(defn index-of
  "finds the index of the first matching element in an array
 
   (index-of even? [1 2 3 4]) => 1
 
   (index-of keyword? [1 2 :hello 4]) => 2"
  {:added "3.0"}
  ([pred coll]
   (loop [[x & more :as coll] coll
          i 0]
     (cond (empty? coll) -1

           (pred x) i

           :else
           (recur more (inc i))))))

(defn element-of
  "finds the element within an array
 
   (element-of keyword? [1 2 :hello 4])
   => :hello"
  {:added "3.0"}
  ([pred coll]
   (loop [[x & more :as coll] coll]
     (cond (empty? coll) nil

           (pred x) x

           :else
           (recur more)))))

(defn flatten-all
  "flattens all elements the collection
 
   (flatten-all [1 2 #{3 {4 5}}])
   => [1 2 3 4 5]"
  {:added "3.0"}
  ([x]
   (filter (complement coll?)
           (rest (tree-seq coll? seq x)))))

(defn all-unique?
  "checks if elements in the collection are unique
 
   (all-unique? [1 2 3 4])
   => true
 
   (all-unique? [1 2 1 4])
   => false"
  {:added "3.0"}
  ([coll]
   (= (count (set coll))
      (count coll))))

(defn conj-unique
  "assumes vector is a set, adding only unique elements
 
   (conj-unique [1 2 3] 1)
   => [1 2 3]
 
   (conj-unique [1 2 3] 4)
   => [1 2 3 4]"
  {:added "3.0"}
  ([coll x]
   (if (neg? (index-of #(= % x) coll))
     (conj coll x)
     coll)))

(defn seqify
  "converts an individual element into a sequence
 
   (seqify 1)
   => [1]
 
   (seqify [1 2 3])
   => [1 2 3]"
  {:added "3.0"}
  ([x]
   (if (sequential? x)
     x
     (list x))))

(defn move-swap
  "swaps items in list around
 
   (move-swap [1 2 3 4] 2 3)
   => [1 2 4 3]"
  {:added "3.0"}
  ([coll i j]
   (let [start (conj (subvec coll 0 i) (nth coll j))
         mid   (conj (subvec coll (inc i) j) (nth coll i))
         end   (subvec coll (inc j) (count coll))]
     (vec (concat start mid end)))))

(defn move-up?
  "checks if item at index `i` can move up
 
   (move-up? [1 2 3 4] 0)
   => false"
  {:added "3.0"}
  ([coll i]
   (move-up? coll i 1))
  ([coll i step]
   (let [len (count coll)]
     (or (not (neg? (- i step)))
         (>= i len)))))

(defn move-up
  "move item at index `i` up `step` positions
 
   (move-up [1 2 3 4] 3)
   => [1 2 4 3]"
  {:added "3.0"}
  ([coll i]
   (move-up coll i 1))
  ([coll i step]
   (if (move-up? coll i step)
     (let [i0    (- i step)
           start (conj (subvec coll 0 i0) (nth coll i))
           mid   (subvec coll i0 i)  
           end   (subvec coll (inc i) (count coll))]
       (vec (concat start mid end)))
     (throw (ex-info "Invalid operation" {:length (count coll)
                                          :i i
                                          :step step})))))

(defn move-down?
  "checks if item at index `i` can move down
 
   (move-down? [1 2 3 4] 0)
   => true"
  {:added "3.0"}
  ([coll i]
   (move-down? coll i 1))
  ([coll i step]
   (let [len (count coll)]
     (and (< i (dec len))
          (not (neg? i))))))

(defn move-down
  "move item at index `i` down `step` positions
 
   (move-down [1 2 3 4] 0)
   => [2 1 3 4]"
  {:added "3.0"}
  ([coll i]
   (move-down coll i 1))
  ([coll i step]
   (if (move-down? coll i step)
     (let [i1  (+ i step)
           start (subvec coll 0 i)
           mid   (conj (subvec coll (inc i) (inc i1)) (nth coll i))
           end   (subvec coll (inc i1) (count coll))]
       (vec (concat start mid end)))
     (throw (ex-info "Invalid operation" {:length (count coll)
                                          :i i
                                          :step step})))))

(defn move?
  "can move item from one index to the other
 
   (move? [1 2 3 4] 1 3)
   => true"
  {:added "3.0"}
  ([coll from to]
   (let [len (count coll)]
     (and (< -1 from len)
          (< -1 to len)))))

(defn move
  "moves item from one index to the other
 
   (move [1 2 3 4] 1 3)
   => [1 3 4 2]"
  {:added "3.0"}
  ([coll from to]
   (cond (= from to)
         coll

         (< from to)
         (move-down coll from (- to from))

         (> from to)
         (move-up coll from (- from to)))))

(defn merge-sorted
  "merges a series of arrays together
 
   (merge-sorted [[1 3 4 6 9 10 15]
                  [2 3 6 7 8 9 10]
                  [1 3 6 7 8 9 10]
                  [1 2 3 4 8 9 10]])
   => [1 1 1 2 2 3 3 3 3 4 4 6 6 6 7 7 8 8 8 9 9 9 9 10 10 10 10 15]"
  {:added "3.0"}
  ([coll]
   (merge-sorted coll identity))
  ([coll key-fn]
   (merge-sorted coll identity <))
  ([coll key-fn comp-fn]
   (->> coll
        (filter seq)
        (u/unfold (fn [s]
                   (if (seq s)
                     (let [[[mf & mn] r]
                           (reduce (fn [[m r] x]
                                     (if (comp-fn (key-fn (first x)) (key-fn (first m)))
                                       [x (cons m r)]
                                       [m (cons x r)]))
                                   [(first s) ()]
                                   (rest s))]
                       (list mf (if mn (cons mn r) r)))))))))
