(ns hara.data.base.template
  (:require [clojure.walk :as walk]
            [hara.util :as u]))

(def ^:dynamic *locals*)

(defn replace-template
  "replace arguments with values
 
   (replace-template '(+ $0 $1 $2) '[x y z])
   => '(+ x y z)"
  {:added "3.0"}
  ([form args]
   (let [m (zipmap
            (map (fn [i] (symbol (str "$" i)))  (range))
            args)]
     (walk/postwalk-replace m form))))

(defmacro get-locals
  "gets local variable symbol names and values
 
   (let [x 1 y 2]
     (get-locals))
   => '{x 1, y 2}"
  {:added "3.0"}
  ([]
   (into {} (for [k (keys &env)]
              [(list 'symbol (name k)) k]))))

(defn local-eval
  "evals local variables
 
   (let [x 1]
     (binding [*locals* (get-locals)]
       (local-eval '(+ x x))))
   => 2"
  {:added "3.0"}
  ([form]
   (local-eval form *locals*))
  ([form locals]
   (eval (list 'let (vec (apply concat
                                (map (fn [k]
                                       [k `(get *locals* (symbol ~(name k)))])
                                     (keys locals))))
               form))))

(defn replace-unquotes
  "replace unquote and unquote splicing
 
   (let [x 1]
     (binding [*locals* (get-locals)]
       (replace-unquotes '(+ ~x ~@(list x 1 2 3)))))
   => '(+ 1 1 1 2 3)"
  {:added "3.0"}
  [form]
  (let [inputs (volatile! [])
        push-fn  (fn [entry]
                   (let [i (count @inputs)]
                     (vswap! inputs conj entry)
                     (symbol (str "$" i))))
        form   (walk/postwalk (fn [form]
                                (cond (not (list? form))
                                      form
                                      
                                      (= (first form) 'clojure.core/unquote)
                                      (push-fn {:tag :unquote :form (second form)})
                                      
                                      (= (first form) 'clojure.core/unquote-splicing)
                                      (push-fn {:tag :unquote-splicing :forms (second form)})

                                      :else
                                      form))
                              form)
        processed (zipmap (map (fn [i] (symbol (str "$" i)))
                               (range))
                          (mapv local-eval @inputs))
        unquote  (->> (u/filter-vals #(-> % :tag (= :unquote)) processed)
                      (u/map-vals :form))
        form     (walk/postwalk-replace unquote form)
        splicing (->> (u/filter-vals #(-> % :tag (= :unquote-splicing)) processed)
                      (u/map-vals :forms))
        form      (walk/postwalk (fn [form]
                                   (cond (not (list? form))
                                         form

                                         :else
                                         (reduce (fn [form e]
                                                   (concat form (or (get splicing e)
                                                                    [e])))
                                                 ()
                                                 form)))
                                 form)]
    form))

(defn template-fn
  "replace body with template
 
  (let [-x- 'a]
     (binding [*locals* (get-locals)]
       (template-fn '(+ ~-x- 2 nil))))
   => '(+ a 2 nil)"
  {:added "3.0"}
  [form]
  (-> form
      (replace-unquotes)))

(defmacro template
  "template macro
 
   (def -x- 'a)
   
   (template (+ ~-x- 2 3))
   => '(+ a 2 3)"
  {:added "3.0"}
  [form]
  `(binding [*locals* (get-locals)]
     (-> (quote ~form)
         (replace-unquotes))))
