(ns hara.io.concurrent.executor
  (:require [hara.protocol.component :as protocol.component]
            [hara.protocol.executor :as protocol.executor]
            [hara.io.concurrent.queue :as q])
  (:import (java.util.concurrent Executors
                                 ExecutorService
                                 ThreadPoolExecutor
                                 ScheduledThreadPoolExecutor
                                 TimeUnit
                                 BlockingQueue)
           (hara.io.concurrent LimitedQueue)))

(def ^:dynamic *track* true)

(defonce ^:dynamic *tracked* (atom #{}))

(defn- ^ExecutorService track [executor]
  (if *track*
    (swap! *tracked* conj executor))
  executor)

(defn- ^ExecutorService untrack [executor]
  (swap! *tracked* disj executor)
  executor)

(defn ^BlockingQueue raw-queue
  "contructs a raw queue in different ways
 
   (cc/raw-queue)
 
   (cc/raw-queue 1)
 
   (cc/raw-queue {:size 1})
 
   (cc/raw-queue {})
 
   (cc/raw-queue (q/queue))"
  {:added "3.0"}
  ([]
   (raw-queue nil))
  ([arg]
   (cond (q/queue? arg) arg

         (nil? arg) (q/queue)

         (integer? arg) (q/fixed-queue arg)
         
         (map? arg)
         (cond (integer? (:size arg))
               (q/fixed-queue (:size arg))

               :else
               (q/queue))

         :else (throw (ex-info "Invalid input" {:input arg})))))

(defn ^ExecutorService single-executor
  "constructs a single executor
 
   ;; any sized pool
   (cc/single-executor)
 
   ;; fixed pool
   (cc/single-executor {:size 10})"
  {:added "3.0"}
  ([]
   (single-executor (q/queue)))
  ([size-or-queue]
   (track (ThreadPoolExecutor. 0 1 0 TimeUnit/MILLISECONDS (raw-queue size-or-queue)))))

(defn ^ExecutorService pool-executor
  "constructs a pool executor
 
   (cc/pool-executor 10 10 1000 {:size 10})"
  {:added "3.0"}
  ([size max keep-alive]
   (pool-executor size max keep-alive (q/queue)))
  ([^long size ^long max ^long keep-alive size-or-queue]
   (track (ThreadPoolExecutor. size max keep-alive TimeUnit/MILLISECONDS (raw-queue size-or-queue)))))

(defn ^ExecutorService cached-executor
  "creates a cached executor
 
   (cc/cached-executor)"
  {:added "3.0"}
  ([]
   (track (Executors/newCachedThreadPool))))

(defn shutdown
  "shuts down executor
 
   (doto (cc/single-executor)
     (cc/shutdown))
   => cc/shutdown?"
  {:added "3.0"}
  ([^ExecutorService service]
   (.shutdown service)))

(defn shutdown-now
  "shuts down executor immediately
 
   (doto (cc/single-executor)
     (cc/shutdown-now))
   => cc/shutdown?"
  {:added "3.0"}
  ([^ExecutorService service]
   (.shutdownNow service)))

(defn ^BlockingQueue get-queue
  "gets the queue from the executor
 
   (-> (cc/pool-executor 10 10 1000 (q/queue))
       (cc/get-queue))"
  {:added "3.0"}
  ([service]
   (if (= (type service) ThreadPoolExecutor)
     (.getQueue ^ThreadPoolExecutor service)
     (throw (ex-info "Cannot access queue" {})))))

(defn ^Callable wrap-min-time
  "constructs a function with a mininum running time
 
   ((cc/wrap-min-time (fn []) 100))"
  {:added "3.0"}
  ([f total]
   (wrap-min-time f total nil))
  ([f total delay]
   (fn []
     (let [start (System/currentTimeMillis)
           _ (if delay (Thread/sleep delay))
           _ (f)
           end (System/currentTimeMillis)
           duration (- end start)]
       (if (> total duration)
         (Thread/sleep (- total duration)))))))

(defn submit
  "submits a task to an executor
 
   @(cc/submit (cc/single-executor)
               (fn [])
               {:min 100})^hidden
   
 
   @(cc/submit (cc/single-executor)
               (fn []
                 (Thread/sleep 1000))
               {:max 100})
  => (throws java.util.concurrent.CancellationException)"
  {:added "3.0"}
  ([^ExecutorService service ^Callable f]
   (submit service f nil))
  ([^ExecutorService service f {:keys [min max]}] 
   (let [^Callable f (cond-> f
                       min (wrap-min-time min))]
     (if max
       (first (.invokeAll service [f] max TimeUnit/MILLISECONDS))
       (.submit service f)))))

(defn submit-notify
  "submits a task (generally to a fixed size queue)
 
   (doto (cc/single-executor 1)
     (cc/submit-notify (fn [])
                      1000)
     (cc/submit-notify (fn [])
                       1000)
     (cc/submit-notify (fn [])
                       1000))"
  {:added "3.0"}
  ([^ExecutorService service ^Callable f]
   (submit-notify service f nil))
  ([^ExecutorService service f {:keys [min max] :as time}]
   (if (pos? (.remainingCapacity (get-queue service)))
     (try (submit service f time)
          (catch Throwable t)))))

(defn ^ScheduledThreadPoolExecutor scheduled-executor
  "constructs a scheduled executor
 
   (cc/scheduled-executor 10)"
  {:added "3.0"}
  ([^long size]
   (ScheduledThreadPoolExecutor. size)))

(defn schedule
  "schedules task for execution"
  {:added "3.0"}
  ([^ScheduledThreadPoolExecutor service ^Callable f ^long interval]
   (schedule service f interval nil))
  ([^ScheduledThreadPoolExecutor service ^Callable f ^long interval {:keys [min]}] 
   (let [^Callable f (cond-> f
                       min (wrap-min-time min))]
     (.schedule service f interval TimeUnit/MILLISECONDS))))

(defn schedule-fixed-rate
  "schedules task at a fixed rate"
  {:added "3.0"}
  ([^ScheduledThreadPoolExecutor service ^Runnable f ^long interval]
   (schedule-fixed-rate service f interval nil))
  ([^ScheduledThreadPoolExecutor service ^Runnable f ^long interval {:keys [initial min]}] 
   (let [^Runnable f (cond-> f
                       min (wrap-min-time min))]
     (.scheduleAtFixedRate service f (or initial 0) interval TimeUnit/MILLISECONDS))))

(defn schedule-fixed-delay
  "schedules task at fixed delay"
  {:added "3.0"}
  ([^ScheduledThreadPoolExecutor service ^Runnable f ^long delay]
   (schedule-fixed-delay service f delay nil))
  ([^ScheduledThreadPoolExecutor service ^Runnable f ^long delay {:keys [initial min]}] 
   (let [^Runnable f (cond-> f
                       min (wrap-min-time min))]
     (.scheduleWithFixedDelay service f (or initial 0) delay TimeUnit/MILLISECONDS))))

(defn await-termination
  "await termination for executor service"
  {:added "3.0"}
  ([^ExecutorService service]
   ((await-termination service Long/MAX_VALUE)))
  ([^ExecutorService service ^long ms]
   (.awaitTermination service ms TimeUnit/MILLISECONDS)))

(defn shutdown?
  "checks if executor is shutdown"
  {:added "3.0"}
  ([^ExecutorService service]
   (.isShutdown service)))

(defn terminated?
  "checks if executor is shutdown and all threads have finished"
  {:added "3.0"}
  ([^ExecutorService service]
   (.isTerminated service)))

(defn terminating?
  "check that executor is terminating"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.isTerminating service)))

(defn current-size
  "returns number of threads in pool"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getPoolSize service)))

(defn current-active
  "returns number of active threads in pool"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getActiveCount service)))

(defn current-submitted
  "returns number of submitted tasks"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getTaskCount service)))

(defn current-completed
  "returns number of completed tasks"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getCompletedTaskCount service)))

(defn pool-size
  "returns the core pool size"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getCorePoolSize service)))

(defn set-pool-size
  "sets the core pool size"
  {:added "3.0"}
  ([^ThreadPoolExecutor service size]
   (.setCorePoolSize service size)))

(defn pool-max
  "returns the core pool max"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getMaximumPoolSize service)))

(defn set-pool-max
  "sets the core pool max"
  {:added "3.0"}
  ([^ThreadPoolExecutor service size]
   (.setMaximumPoolSize service size)))

(defn keep-alive
  "returns the keep alive time"
  {:added "3.0"}
  ([^ThreadPoolExecutor service]
   (.getKeepAliveTime service TimeUnit/MILLISECONDS)))

(defn set-keep-alive
  "sets the keep alive time"
  {:added "3.0"}
  ([^ThreadPoolExecutor service length]
   (.setKeepAliveTime service length TimeUnit/MILLISECONDS)))

(defn set-rejected-handler
  "sets the rejected task handler"
  {:added "3.0"}
  [^ThreadPoolExecutor service f]
  (.setRejectedExecutionHandler service f))

(defn at-capacity?
  "checks if executor is at capacity"
  {:added "3.0"}
  ([executor]
   (= (current-active executor)
      (pool-size executor))))

(defn increase-capacity
  "increases the capacity of the executor"
  {:added "3.0"}
  ([executor]
   (increase-capacity executor (* 2 (pool-size executor))))
  ([executor n]
   (let [max (pool-max executor)]
     (if (< max n) (set-pool-max executor n))
     (set-pool-size executor n))))

(defn raw-executor-type
  "returns executor service type"
  {:added "3.0"}
  [^ThreadPoolExecutor executor]
  (cond (instance? ScheduledThreadPoolExecutor executor)
        :scheduled

        (= 1 (pool-max executor))
        :single

        (and (= Integer/MAX_VALUE (pool-max executor))
             (let [queue (.getQueue executor)]
               (and (zero? (q/remaining-capacity queue))
                    (zero? (count queue)))))
        :cached

        :else :pool))

(defn raw-executor-info
  "returns executor service info"
  {:added "3.0"}
  ([^ThreadPoolExecutor executor]
   (raw-executor-info executor #{:type :running :current :counter :options}))
  ([^ThreadPoolExecutor executor k]
   (let [[return items]  (cond (keyword? k)
                               [k #{k}]
                               
                               :else
                               [identity (set k)])
         queue  (.getQueue executor)]
     (cond-> {}
       (:type items)    (assoc :type (raw-executor-type executor))
       (:running items) (assoc :running (not (shutdown? executor)))
       (:current items) (assoc :current {:threads (current-size executor)
                                      :active  (current-active executor)
                                      :queued  (count queue)
                                      :terminated (terminated? executor)})
       (:counter items) (assoc :counter {:submit   (current-submitted executor)
                                      :complete (current-completed executor)})
       (:options items) (assoc :options {:pool {:size (pool-size executor)
                                             :max (pool-max executor)
                                             :keep-alive (keep-alive executor)}
                                      :queue {:remaining (q/remaining-capacity queue)
                                              :total (count queue)}})
       :then return))))

(defn raw-executor-props
  "returns props for getters and setters"
  {:added "3.0"}
  ([^ThreadPoolExecutor executor]
   {:pool {:size {:get pool-size
                  :set set-pool-size}
           :max  {:get pool-max
                  :set set-pool-max}
           :keep-alive {:get keep-alive
                        :set set-keep-alive}}}))

(extend-type java.util.concurrent.ThreadPoolExecutor
  protocol.component/IComponent
  (-start [exe] exe)
  (-started [exe] (not (shutdown? exe)))
  (-stop  [exe] (doto exe (shutdown)))
  (-info  [exe level] (raw-executor-info exe))
  (-props [exe] (raw-executor-props exe))
  
  protocol.executor/IExecutor
  (-submit     [executor callable]
    (submit executor ^Callable callable))
  
  (-force-kill [executor]
    (shutdown-now executor)))

(defmethod print-method ThreadPoolExecutor
  [v ^java.io.Writer w]
  (.write w (str "#raw.executor" (raw-executor-info v))))

(defmethod print-method ScheduledThreadPoolExecutor
  [v ^java.io.Writer w]
  (.write w (str "#scheduled.executor" (raw-executor-info v))))

(defmulti raw-executor
  "creates an executor
 
   (cc/raw-executor {:type :pool
                 :size 3
                 :max 3
                 :keep-alive 1000})
 
   (cc/raw-executor {:type :single
                 :size 1})"
  {:added "3.0"}
  :type)

(defmethod raw-executor :pool
  ([{:keys [size max keep-alive queue] :as m}]
   (pool-executor size max keep-alive queue)))

(defmethod raw-executor :single
  ([{:keys [queue] :as m}]
   (single-executor queue)))

(defmethod raw-executor :scheduled
  ([{:keys [size] :as m}]
   (scheduled-executor size)))

(defmethod raw-executor :cached
  ([_]
   (cached-executor)))

(defn tracked-executors
  ([] @*tracked*))

(defn active-executors
  ([] (remove terminated? @*tracked*)))

