(ns hara.module.base.generator
  (:require [hara.util :as u]
            [clojure.set :as set]))

(def ^:dynamic *self* false)

(defn split-body
  "splits a body depending on keyword pairs
 
   (split-body [:a 1 :b 2 '[hello] '[there]])
   => [{:a 1, :b 2} '([hello] [there])]"
  {:added "3.0"}
  ([body]
   (let [params (->> (partition 2 body)
                     (take-while (comp keyword? first))
                     (map vec))
         len (* 2 (count params))
         body (drop len body)]
     [(into {} params) body])))

(defn split-single
  "splits out a given entry"
  {:added "3.0"}
  ([forms]
   (split-single forms :tag))
  ([forms tag-key]
   (let [ptl (first forms)
         [params body] (split-body (rest forms))]
     [(assoc params tag-key ptl) body])))

(defn split-all
  "splits all entries
 
   (split-all '[ITest
                :include [-test]
                :body {-val 3}
                IMore
                IProtocols
                :exclude []]
              :protocol
              {:prefix \"test/\"})
  => '[{:prefix \"test/\", :include [-test], :body {-val 3}, :protocol ITest}
        {:prefix \"test/\", :protocol IMore}
        {:prefix \"test/\", :exclude [], :protocol IProtocols}]"
  {:added "3.0"}
  ([forms]
   (split-all forms :tag))
  ([forms tag-key]
   (split-all forms tag-key {}))
  ([forms tag-key params]
   (loop [forms forms
          acc []]
     (let [[ptl more] (split-single forms tag-key)
           ptl (merge params ptl)
           acc (conj acc ptl)]
       (if (empty? more)
         acc
         (recur more acc))))))

(defn unwrap-sym
  "unwraps the protocol symbol
 
   (unwrap-sym {:name '-val :prefix \"test/\" :suffix \"-mock\"})
   => 'test/val-mock"
  {:added "3.0"}
  ([{:keys [name prefix suffix]
     :or {prefix "" suffix ""}}]
   (symbol (str prefix (subs (u/keystring name) 1) suffix))))

(defn wrap-sym
  "wraps the protocol symbol
 
   (wrap-sym {:protocol 'protocol.test/ITest :name '-val})
   => 'protocol.test/-val"
  {:added "3.0"}
  ([{:keys [protocol name]}]
   (symbol (namespace protocol) (u/keystring name))))

(defn standard-body-input-fn
  "creates a standard input function"
  {:added "3.0"}
  ([_]
   (fn [{:keys [arglist]}] arglist)))

(defn standard-body-output-fn
  "creates a standard output function"
  {:added "3.0"}
  ([{:keys [body-sym-fn body-arg-fn]
     :or {body-sym-fn :name
          body-arg-fn identity}}]
   (fn [{:keys [arglist] :as signature}]
     (let [body `(~(body-sym-fn signature)
                  ~(body-arg-fn (first arglist))
                  ~@(rest arglist))]
       (if *self*
         `(do ~body ~(first arglist))
         body)))))

(defn create-body-fn
  "creates a body function
 
   ((create-body-fn {:body-sym-fn unwrap-sym})
    {:name '-val :arglist '[cache val] :prefix \"test/\"})
   => '([cache val] (test/val cache val))"
  {:added "3.0"}
  [{:keys [body-fn] :as fns}]
  (cond body-fn
        body-fn

        :else
        (let [{:keys [body-input-fn body-output-fn]
               :or {body-input-fn  (standard-body-input-fn fns)
                    body-output-fn (standard-body-output-fn fns)}} fns]
          (fn [signature]
            `(~(body-input-fn signature) ~(body-output-fn signature))))))

(defn template-signatures
  "finds template signatures for a protocol"
  {:added "3.0"}
  ([protocol]
   (template-signatures protocol {}))
  ([protocol params]
   (let [sigs (if-let [var (-> protocol resolve)]
                (-> var deref :sigs)
                (throw (ex-info "Cannot find protocol" {:input protocol})))]
     (mapcat (fn [{:keys [arglists] :as m}]
               (let [m (-> (merge params m)
                           (dissoc :arglists))]
                 (map (fn [arglist]
                        (assoc m :arglist arglist))
                      arglists)))
             (vals sigs)))))

(defn template-transform
  "transforms all functions"
  {:added "3.0"}
  ([signatures {:keys [template-fn] :as fns
                :or {template-fn identity}}]
   (let [body-fn  (create-body-fn fns)
         sym-fn   (fn [{:keys [name] :as m}] (with-meta name m))
         all (map (juxt sym-fn body-fn)
                  signatures)]
     (template-fn all))))

(defn parse-impl
  "parses the different transform types
 
   (parse-impl (template-signatures 'IHello {:prefix \"impl/\"})
               {:body '{-hi \"Hi There\"}})
   => '{:method #{},
        :body #{-hi},
        :default #{-delete}}"
  {:added "3.0"}
  ([signatures {:keys [method body include exclude] :as params}]
   (let [include-syms (if include (set include))
         exclude-syms (set exclude)
         method-syms  (set (keys method))
         body-syms    (set (keys body))
         default-syms (set/difference (or include-syms
                                          (set (map :name signatures)))
                                      (set/union method-syms
                                                 body-syms
                                                 exclude-syms))]
     {:method method-syms
      :body body-syms
      :default default-syms})))


(comment
  (use 'hara.code)
  (hara.code/scaffold)
  

  (parse-impl (template-signatures 'hara.protocol.cache/ICache {:prefix "impl/"})
              {:exclude '[-export -import]
               :body  '{-has? true}})

  (template-transform
   (filter #(-> % :name (= '-has?))
           (template-signatures 'hara.protocol.cache/ICache {:prefix "impl/"}))
   (merge (impl-fns :body '{-has? true})
          {:template-fn dimpl-template-fn}))

  (comment
    (template-transform
     (template-signatures 'hara.protocol.cache/ICache {:prefix "impl/"})
     {:body-sym-fn unwrap-sym
      ;;:body-output-fn (fn [_] nil)
      :template-fn dimpl-single-template})
    
    (def +opts+ {:prefix "atom/"
                 :suffix "-atom"
                 :protocol 'protocol.cache/ICache
                 :name '-get
                 :arglist '[cache more]}))
  
  (comment
    (->> (map (fn [{:keys [protocol] :as params}]
                [protocol (template-signatures protocol params)])
              (split-all '[ITest] :protocol {:prefix "common/"}))
         (into {}))
    
    ;;(body-transform +opts+ {:body-sym-fn unwrap-sym})
    ;;(body-transform +opts+ {:body-sym-fn wrap-sym-fn})


    (:sigs ITest)
    ;;{:-val {:variadic true, :notify true, :name -val, :arglists ([cache]), :doc nil}}

    )
  )
