(ns hara.module.base.impl
  (:require [hara.util :as u]
            [clojure.set :as set]
            [clojure.walk :as walk]
            [hara.module.base.generator :as gen]
            [hara.data.base.nested :as nested]
            [hara.module.base.include :as include]))

(include/include
 (hara.module.base.generator wrap-sym
                             unwrap-sym))

(defn template-gen
  "generates forms given various formats
 
   (template-gen protocol-fns [:default]
                 (gen/template-signatures 'ITest) {} {})
   => '([-val ([obj] (val obj))]
        [-val ([obj k] (val obj k))]
        [-get ([obj] (get obj))])"
  {:added "3.0"}
  [type-fn types signatures params fns]
  (let [syms (gen/parse-impl signatures params)]
    (mapcat (fn [type]
              (let [fns (merge (type-fn type (get params type)) fns)]
                (u/->> signatures
                       (filter (fn [{:keys [name]}]
                                 ((get syms type) name)))
                       (gen/template-transform % (or (get fns type)
                                                     fns)))))
            types)))

(defn protocol-fns
  "helpers for protocol forms
 
   (protocol-fns :body {})
   => (contains {:body-output-fn fn?})"
  {:added "3.0"}
  ([type template]
   (case type
     :default {:body-sym-fn gen/unwrap-sym}
     :method  {:body-sym-fn (fn [{:keys [name]}] (get template name))}
     :body    {:body-output-fn (fn [{:keys [name]}] (get template name))})))

(defn dimpl-template-fn
  "helper function for defimpl"
  {:added "3.0"}
  ([inputs]
   (map (fn [[name body]]
          (apply list name body))
        inputs)))

(defn dimpl-template-protocol
  "coverts the entry into a template
 
   (dimpl-template-protocol {:protocol 'ITest :prefix \"impl/\" :suffix \"-test\"})
   => '(ITest (-val [obj] (impl/val-test obj))
              (-val [obj k] (impl/val-test obj k))
              (-get [obj] (impl/get-test obj)))"
  {:added "3.0"}
  ([{:keys [protocol prefix suffix] :as params}]
   (let [signatures (gen/template-signatures protocol {:prefix prefix :suffix suffix})
         types [:default :method :body]]
     (->> (template-gen protocol-fns types signatures params {:template-fn dimpl-template-fn})
          (cons protocol)))))

(defn interface-fns
  "helper for interface forms
 
   (interface-fns :body {})
   => (contains {:body-output-fn fn?})"
  {:added "3.0"}
  ([type template]
   (case type
     :method  {:body-sym-fn (fn [{:keys [name arglist]}] (get-in template [name arglist]))}
     :body    {:body-output-fn (fn [{:keys [name arglist]}] (get-in template [name arglist]))})))

(defn dimpl-template-interface
  "creates forms for the interface
 
   (dimpl-template-interface {:interface 'ITest
                              :method '{invoke {[entry] submit-invoke}}
                              :body   '{bulk? {[entry] false}}})
   => '(ITest (invoke [entry] (submit-invoke entry))
              (bulk? [entry] false))"
  {:added "3.0"}
  ([{:keys [interface method body] :as params}]
   (let [all  (merge-with merge method body)
         signatures (mapcat (fn [[name methods]]
                              (map (fn [[args _]]
                                     (merge {:arglist args
                                             :name name}
                                            (dissoc params :method body)))
                                   methods))
                            all)
         body-sym    (fn [{:keys [name arglist] :as params}]
                       (get-in method [name arglist]))
         body-output (fn [{:keys [name arglist] :as params}]
                       (get-in body [name arglist]))
         types [:method :body]]
     (->> (template-gen interface-fns types signatures params {:template-fn dimpl-template-fn})
          (cons interface)))))

(defn dimpl-print-method
  "creates a print method form"
  {:added "3.0"}
  [sym]
  `(defmethod print-method (Class/forName ~(str *ns* "." sym))
     [~'v ~(with-meta 'w {:tag 'java.io.Writer})]
     (.write ~'w (str ~'v))))

(defn dimpl-form
  "helper for `defimpl`"
  {:added "3.0"}
  ([sym bindings body]
   (let [[params body] (gen/split-body body)
         {:keys [type prefix suffix string protocols interfaces]
          :or {type 'defrecord prefix "" suffix ""}} params
         protocols   (gen/split-all protocols :protocol {:prefix prefix :suffix suffix})
         protocol-forms (mapcat dimpl-template-protocol protocols)
         interfaces  (if string
                       (concat ['Object :method {'toString {'[obj] string}}] interfaces)
                       interfaces)
         interfaces  (gen/split-all interfaces :interface {:prefix prefix :suffix suffix})
         interfaces-forms (mapcat dimpl-template-interface interfaces)]
     `[(~type ~sym ~bindings ~@(concat protocol-forms interfaces-forms))
       ~@(if string
           [(dimpl-print-method sym)]
           [])])))

(defmacro defimpl
  "creates a high level `deftype` or `defrecord` interface"
  {:added "3.0"}
  ([sym bindings & body]
   (dimpl-form sym bindings body)))

;;;;
;;;;
;;;;

(defn eimpl-template-fn
  "creates forms compatible with `extend-type` and `extend-protocol`
 
   (eimpl-template-fn '([-val ([obj] (val obj))]
                        [-val ([obj k] (val obj k))]
                        [-get ([obj] (get obj))]))
   => '((-val ([obj] (val obj))
              ([obj k] (val obj k)))
        (-get ([obj] (get obj))))"
  {:added "3.0"}
  ([inputs]
   (->> (group-by first inputs)
        (map (fn [[name arr]]
               `(~name ~@(map second arr)))))))

(defn eimpl-template-protocol
  "helper for eimpl-form
 
   (eimpl-template-protocol {:protocol 'ITest :prefix \"impl/\" :suffix \"-test\"})
   => '(ITest
        (-val ([obj] (impl/val-test obj))
              ([obj k] (impl/val-test obj k)))
        (-get ([obj] (impl/get-test obj))))"
  {:added "3.0"}
  ([{:keys [protocol prefix suffix] :as params}]
   (let [signatures (gen/template-signatures protocol {:prefix prefix :suffix suffix})
         types [:default :method :body]]
     (cons protocol (template-gen protocol-fns types signatures params {:template-fn eimpl-template-fn})))))

(defn eimpl-form
  "creates the extend-impl form"
  {:added "3.0"}
  ([class body]
   (let [[params _] (gen/split-body body)
         {:keys [prefix suffix string protocols]
          :or {prefix "" suffix ""}} params
         protocols   (gen/split-all protocols :protocol {:prefix prefix :suffix suffix :class class})
         protocol-forms (mapcat eimpl-template-protocol protocols)]
     `(extend-type ~class ~@protocol-forms))))

(defmacro extend-impl
  "extends a class with the protocols"
  {:added "3.0"}
  ([type & body]
   (eimpl-form type body)))

;;;;
;;;;
;;;;

(defn build-variadic-fn
  ([fsym arr]
   (let [[len i] (->> (map-indexed (fn [i {:keys [arglist]}]
                                     [(count arglist) i])
                                   arr)
                      (sort)
                      (last))
         [_ [arglist & body]] (get arr i)
         arglist (vec (concat (butlast arglist)
                              ['& (last arglist)]))
         oarr (keep-indexed (fn [j e] (if (not= j i)
                                        e))
                            arr)]
     `(defn ~fsym
        ~@(map second oarr)
        (~arglist ~@body)))))

(defn build-template-fn
  "contructs a template from returned vals with support for variadic
 
   ((build-template-fn {}) '([-val ([obj] (val obj))]
                             [-val ([obj k] (val obj k))]
                             [-get ([obj] (get obj))]))
   => '((clojure.core/defn val
          ([obj] (val obj))
          ([obj k] (val obj k)))
        (clojure.core/defn get ([obj] (get obj))))"
  {:added "3.0"}
  ([{:keys [variadic] :as opts}]
   (fn [inputs]
     (->> (group-by first inputs)
          (map (fn [[name arr]]
                 (let [fsym (gen/unwrap-sym (assoc opts :name name))]
                   (if (and variadic (variadic name))
                     (build-variadic-fn fsym arr)
                     `(defn ~fsym ~@(map second arr))))))))))

(defn build-template-protocol
  "helper for build
 
   (build-template-protocol '{:protocol ITest
                              :outer {:suffix \"-outer\"}
                              :inner {:prefix \"inner-\"}
                              :fns {:body-sym-fn gen/unwrap-sym}})
 
   => '((clojure.core/defn val-outer
          ([obj] (inner-val obj))
          ([obj k] (inner-val obj k)))
        (clojure.core/defn get-outer ([obj] (inner-get obj))))"
  {:added "3.0"}
  ([{:keys [protocol inner outer fns variadic] :as params}]
   (let [variadic (set variadic)
         inner (merge inner {:protocol protocol :variadic variadic})
         outer (merge outer {:protocol protocol :variadic variadic})
         signatures (gen/template-signatures protocol inner)
         types [:default :method :body]
         fns (u/map-vals (fn [val] (cond (symbol? val)
                                         @(resolve val)
                                         
                                         :else
                                         (eval val)))
                         fns)
         template-fn (build-template-fn outer)
         forms (template-gen protocol-fns types signatures params
                             (nested/merge-nested {:template-fn template-fn
                                                   :default {:body-sym-fn gen/wrap-sym
                                                             :template-fn template-fn}}
                                                  fns))]
     forms)))

(defn build-form
  "allows multiple forms to be built"
  {:added "3.0"}
  ([body]
   (let [[global body] (if (map? (first body))
                         [(first body) (rest body)]
                         [{} body])
         protocols  (gen/split-all body :protocol global)
         forms      (mapcat build-template-protocol protocols)]
     forms)))

(defmacro build
  "build macro for generating functions from protocols"
  {:added "3.0" :style/indent 1}
  ([& body]
   (vec (build-form body))))
