(ns hara.print.base.local
  (:require [hara.io.concurrent :as cc]
            [hara.string :as string]
            [hara.util :as u])
  (:refer-clojure :exclude [print println prn with-out-str]))

(def ^:dynamic *local* true)

(def ^:dynamic *indent* 0)

(def ^:dynamic *indent-step* 3)

(def ^:dynamic *defaults* {:interval 50 :max-batch 1000})

(defonce ^:dynamic *queue* (atom []))

(defonce ^:dynamic *executor* (delay (cc/single-executor 1)))

(defn print-format-indent
  "formats the line with indent"
  {:added "3.0"}
  ([output]
   (print-format-indent output *indent*))
  ([output indent]
   (if (or (nil? indent)
           (zero? indent))
     output
     (let [space (apply str (repeat indent " "))]
       (->> output
            (string/split-lines)
            (map (partial str space))
            (string/join "\n"))))))

(defn print-handler
  "handler for local print"
  {:added "3.0"}
  ([]
   (cc/process-atom-queue (fn [items]
                            (doseq [item items]
                              (clojure.core/print item)
                              (flush)))
                          *queue*
                          (:max-batch *defaults*))))

(defn print
  "prints using local handler"
  {:added "3.0"}
  ([& items]
   (if *local*
     (do (swap! *queue* #(into % items))
         (cc/submit-notify @*executor*
                           print-handler
                           (:interval *defaults*))
         nil)
     (apply clojure.core/print items))))

(defn println
  "convenience function for println"
  {:added "3.0"}
  ([& items]
   (print (clojure.core/with-out-str
            (apply clojure.core/println items)))))

(defn prn
  "convenience function for prn"
  {:added "3.0"}
  ([& items]
   (print (clojure.core/with-out-str
            (apply clojure.core/prn items)))))

(defmacro with-system
  "with system print instead of local"
  {:added "3.0"}
  ([& body]
   `(binding [*local* false]
      ~@body)))

(defmacro with-out-str
  "gets the local string
 
   (local/with-out-str (local/print \"hello\"))
   => \"hello\""
  {:added "3.0"}
  ([& body]
   `(binding [*local* false]
      (clojure.core/with-out-str ~@body))))
