(ns hara.util
  (:refer-clojure :exclude [-> ->> keyword *ns* reset! aget require])
  (:require [clojure.java.shell :as sh])
  (:import (hara.core.base Clock Flake Counter)
           (java.util Date)
           (java.net Socket ConnectException)))

(def ^:dynamic *ns* nil)

(def ^:dynamic *native-compile* false)

(defn native?
  ([]
   (.startsWith (System/getProperty "java.vm.name")
                "Substrate VM")))

(defn T
  "returns `true` for any combination of input `args`
 
   (T) => true
   (T :hello) => true
   (T 1 2 3) => true"
  {:added "3.0"}
  [& args] true)

(defn F
  "returns `false` for any combination of input `args`
 
   (F) => false
   (F :hello) => false
   (F 1 2 3) => false"
  {:added "3.0"}
  [& args] false)

(defn NIL
  "returns `nil` for any combination of input `args`
 
   (NIL) => nil
   (NIL :hello) => nil
   (NIL 1 2 3) => nil"
  {:added "3.0"}
  [& args] nil)

(defn queue
  "returns a `clojure.lang.PersistentQueue` object.
 
   (def a (queue 1 2 3 4))
   (pop a) => [2 3 4]"
  {:added "3.0"}
  ([] (clojure.lang.PersistentQueue/EMPTY))
  ([x] (conj (queue) x))
  ([x & xs] (apply conj (queue) x xs)))

(defmethod print-method clojure.lang.PersistentQueue
  [v ^java.io.Writer w]
  (.write w (str (into [] v))))

(defn sid
  "returns a short id string
 
   (sid)
   ;; \"t8a6euzk9usy\"
   => string?"
  {:added "3.0"}
  []
  (clojure.core/->
   (str (java.util.UUID/randomUUID))
   (.getBytes)
   (java.nio.ByteBuffer/wrap)
   (.getLong)
   (Long/toString Character/MAX_RADIX)))

(defn uuid
  "returns a `java.util.UUID` object
 
   (uuid) => #(instance? java.util.UUID %)
 
   (uuid \"00000000-0000-0000-0000-000000000000\")
   => #uuid \"00000000-0000-0000-0000-000000000000\""
  {:added "3.0"}
  ([] (java.util.UUID/randomUUID))
  ([id]
   (cond (string? id)
         (java.util.UUID/fromString id)

         (instance? (Class/forName "[B") id)
         (java.util.UUID/nameUUIDFromBytes id)

         :else
         (throw (ex-info (str id " can only be a string or byte array")))))
  ([^Long msb ^Long lsb]
   (java.util.UUID. msb lsb)))

(defn instant
  "returns a `java.util.Date` object
 
   (instant) => #(instance? java.util.Date %)
 
   (instant 0) => #inst \"1970-01-01T00:00:00.000-00:00\""
  {:added "3.0"}
  ([] (java.util.Date.))
  ([^Long val] (java.util.Date. val)))

(defn uri
  "returns a `java.net.UrI` object
 
   (uri \"http://www.google.com\")
   => java.net.URI"
  {:added "3.0"}
  [path] (java.net.URI/create path))

(defn url
  "retruns a `java.net.URL` object
 
   (url \"http://www.google.com\")
   => java.net.URL"
  {:added "3.0"}
  [path] (java.net.URL. path))
  
(defn date
  "creates a new date
 
   (date 1000)
   => #inst \"1970-01-01T00:00:01.000-00:00\""
  {:added "3.0"}
  ([]  (Date.))
  ([^long t] (Date. t)))

(defn keystring
  "returns the keystring value
 
   (keystring :hello) => \"hello\"
   
   (keystring \"hello\") => \"hello\""
  {:added "3.0"}
  ^String
  [obj]
  (cond (keyword? obj)
        (subs (str obj) 1)

        (string? obj)
        obj

        :else
        (str obj)))
        
(defn keyword
  "returns the keyword value
 
   (keyword :hello) => :hello
   
   (keyword \"hello\") => :hello"
  {:added "3.0"}
  [obj]
  (cond (keyword? obj)
        obj

        (string? obj)
        (clojure.core/keyword obj)

        :else
        (throw (ex-info "Cannot process obj" {:input obj}))))

(defn string
  "creates a string from a byte array
 
   (string (.getBytes \"Hello\"))
   => \"Hello\""
  {:added "3.0"}
  ([obj]
   (cond (bytes? obj)
         (String. ^bytes obj)

         :else
         (str obj))))

(defn hash-label
  "returns a string repesentation of the object for internally unique keys
 
   (hash-label 1) => \"__1__\"
   (hash-label \"a\" \"b\" \"c\") => \"__97_98_99__\"
   (hash-label \"abc\") => \"__96354__\""
  {:added "3.0"}
  ([^Object obj] (str "__" (.hashCode obj) "__"))
  ([^Object obj & more]
   (let [result (clojure.core/->>
                 (cons obj more)
                 (map (fn [^Object x] (.hashCode x)))
                 (clojure.string/join "_"))]
     (str "__" result "__"))))

(defmacro suppress
  "Suppresses any errors thrown in the body.
 
   (suppress (throw (ex-info \"Error\" {}))) => nil
 
   (suppress (throw (ex-info \"Error\" {})) :error) => :error
 
   (suppress (throw (ex-info \"Error\" {}))
             (fn [^Throwable e]
               (.getMessage e))) => \"Error\""
  {:added "3.0"}
  ([body]
   `(try ~body (catch Throwable ~'t)))
  ([body catch-val]
   `(try ~body (catch Throwable ~'t
                 (cond (fn? ~catch-val)
                       (~catch-val ~'t)
                       :else ~catch-val)))))

(defmacro explode
  ([& body]
   `(try ~@body (catch Throwable ~'t
                  (.printStackTrace ~'t)))))

(defn edn
  "prints then reads back the string
   
   ((juxt type
          (comp type edn)) (symbol \":hello\"))
   => [clojure.lang.Symbol
       clojure.lang.Keyword]"
  {:added "3.0"}
  ([obj]
   (read-string (pr-str obj))))

(defn counter
  "creates a counter
 
   (pr-str (counter))
   => \"#counter{0}\""
  {:added "3.0"}
  ([] (counter 0))
  ([n]
   (Counter. n)))

(defmethod print-method Counter
  [v ^java.io.Writer w]
  (.write w (format "#counter{%d}" @v)))

(defmacro inc!
  "increments the counter
 
   @(doto (counter)
      (inc!)
      (inc! 10))
   => 11"
  {:added "3.0"}
  ([counter]
   `(.inc ~(with-meta counter {:tag 'hara.core.base.Counter})))
  ([counter n]
   `(.inc ~(with-meta counter {:tag 'hara.core.base.Counter}) ~n)))

(defmacro dec!
  "decrements the counter
 
   @(doto (counter 11)
      (dec!)
      (dec! 10))
   => 0"
  {:added "3.0"}
  ([counter]
   `(.dec ~(with-meta counter {:tag 'hara.core.base.Counter})))
  ([counter n]
   `(.dec ~(with-meta counter {:tag 'hara.core.base.Counter}) ~n)))

(defn reset!
  "resets a counter to the given value
 
   (reset! (counter 10) 5)
   => 5"
  {:added "3.0"}
  ([^Counter counter n]
   (.reset counter n)))
  
(defn system-ns
  "returns the system nano time
 
   (system-ns)
   ;; 8158606456270
   => number?"
  {:added "3.0"}
  ([]
   (System/nanoTime)))

(defn system-ms
  "returns the system milli time
 
   (system-ms)
   => number?"
  {:added "3.0"}
  ([]
   (System/currentTimeMillis)))

(defn time-ns
  "returns current time in nano seconds
 
   (time-ns)
   ;; 1593922814991449423
   => number?"
  {:added "3.0"}
  ([]
   (Clock/currentTimeNanos)))

(defn time-us
  "returns current time in micro seconds
 
   (time-us)
   ;; 1593922882412533
   => number?"
  {:added "3.0"}
  ([]
   (Clock/currentTimeMicros)))

(defn time-ms
  "returns current time in milli seconds
 
   (time-ms)
   ;; 1593922917603
   => number?"
  {:added "3.0"}
  ([]
   (Clock/currentTimeMillis)))

(defn format-ms
  "returns ms time is human readable format
 
   (format-ms 10000000)
   => \"02h 46m 40s\""
  {:added "3.0"}
  ([time]
   (let [millis  (mod time  1000)
         seconds (mod (quot time 1000) 60)
         mins    (mod (quot time (* 60 1000)) 60)
         hours   (mod (quot time (* 60 60 1000)) 24)
         days    (quot time (* 24 60 60 1000))
         level   (count (drop-while zero? [days hours mins seconds millis]))]
     (case level
       0 "0ms"
       1 (format "%03dms" millis)
       2 (format "%02ds %03dms" seconds millis)
       3 (format "%02dm %02ds" mins seconds)
       4 (format "%02dh %02dm %02ds" hours mins seconds)
       5 (format "%02dd %02dh %02dm" days mins seconds)))))

(defn parse-ms
  "parses the string representation of time in ms
 
   (parse-ms \"1s\")
   => 1000
 
   (parse-ms \"0.5h\")
   => 1800000"
  {:added "3.0"}
  [^String s]
  (let [len (count s)
        [s unit] (cond (.endsWith s "ms")
                       [(subs s 0 (- len 2)) 1]
                 
                 :else
                 (let [unit (case (last s)
                              \d (* 24 60 60 1000)
                              \h (* 60 60 1000)
                              \m (* 60 1000)
                              \s 1000)]
                   [(subs s 0 (dec len)) unit]))
        v (Double/parseDouble s)]
    (long (* v unit))))

(defn format-ns
  "returns ns time in seconds
 
   (format-ns 1000000)
   => \"1.000ms\""
  {:added "3.0"}
  ([time]
   (format-ns time 4))
  ([time digits]
   (let [full   (str time)
         len    (count full)
         suffix (cond (<= 1 len 3)
                      "ns"
                      (<= 4 len 6)
                      "us"
                      (<= 7 len 9)
                      "ms"
                      (<= 10 len 12)
                      "s"
                      (<= 13 len 15)
                      "ks"
                      (<= 16 len 18)
                      "Ms"
                      (<= 19 len 21)
                      "Gs")
         decimal (inc (rem (dec len) 3))]
     (if (<= len digits)
       (str full "ns")
       (let [out (subs full 0 digits)]
         (str (subs out 0 decimal)
              "."
              (subs out decimal)
              suffix))))))

(defmacro bench-ns
  "measures a block in nanoseconds
 
   (bench-ns (Thread/sleep 1))
   => #(< 500000 % 2000000)"
  {:added "3.0"}
  ([& body]
   `(let [_# (System/gc)
          start# (time-ns)]
      (do ~@body)
      (- (time-ns) start#))))

(defmacro bench-ms
  "measures a block in milliseconds
 
   (bench-ms (Thread/sleep 10))
   => #(< 5 % 20)"
  {:added "3.0"}
  ([& body]
   `(let [_# (System/gc)
          start# (time-ms)]
      (do ~@body)
      (- (time-ms) start#))))

(defn parse-ns
  "parses the string repesentation of time in ns
 
   (parse-ns \"2ns\")
   => 2
 
   (parse-ns \"0.3s\")
   => 300000000"
  {:added "3.0"}
  [^String s]
  (if (.endsWith s "s")
    (let [len  (count s)
          nchar (nth s (- len 2))
          [s unit] (cond (<= (int \0) (int nchar) (int \9))
                         [(subs s 0 (dec len)) 1000000000]
                         
                         :else
                         (let [unit (case nchar
                                      \n 1
                                      \u 1000
                                      \m 1000000
                                      \k 1000000000000
                                      \M 1000000000000000
                                      \G 1000000000000000000)]
                           [(subs s 0 (- len 2)) unit]))
          v (Double/parseDouble s)]
      (long (* v unit)))))

(defn flake
  "returns a unique, time incremental id
 
   (flake)
   ;; \"4Wv-T47LftnVlHhhh00JYuU15NCuNLcr\"
   => string?"
  {:added "3.0"}
  []
  (str (Flake/flake)))

(defn ns-current
  "returns a thread safe *ns*
 
   (ns-current)"
  {:added "3.0"}
  []
  (or *ns* clojure.core/*ns*))

(defn ns-get
  "gets a symbol in the current namespace
 
   (ns-get 'hara.util \"ns-get\")"
  {:added "3.0"}
  ([sym]
   (ns-get (or *ns* clojure.core/*ns*) sym))
  ([ns k]
   (let [elem (resolve (symbol (str ns) (str k)))]
     (if elem
       @elem
       (ex-info (format "Cannot find %s in namespace" k)
                {:ns ns})))))

(defmacro with-ns
  "binds the thread safe *ns* var
 
   (with-ns hara.util
     (ns-get \"ns-get\"))"
  {:added "3.0"}
  [ns & body]
  `(binding [*ns* (quote ~ns)]
     ~@body))

(defmacro this-ns
  "returns the current namespace the code is in
 
   (this-ns)
   => 'hara.util_test"
  {:added "3.0"}
  []
  `(clojure.core/->> (fn []) str (re-find #"^.*?(?=\$|$)") symbol))

(defn thread-form
  "helper function to '->' and '->>'
 
   (thread-form '[(+ 1) dec]
                (fn [form] (concat form ['%])))
   => '(% (+ 1 %) % (dec %))"
  {:added "3.0"}
  [forms transform-fn]
  (interleave (repeat '%)
              (map (fn [form]
                     (cond (list? form)
                           (if (some (fn [v] (= v '%)) (flatten form))
                             form
                             (transform-fn form))

                           :else
                           (transform-fn (list form))))
                   forms)))

(defmacro ->
  "like -> but with % as placeholder
 
   (-> 1
       inc
       (- 10 %))
   => 8"
  {:added "3.0"}
  ([expr & forms]
   `(let [~'% ~expr
          ~@(thread-form forms (fn [form] (concat [(first form) '%] (rest form))))]
      ~'%)))

(defmacro ->>
  "like ->> but with % as placeholder
 
   (-> 1
       inc
       (- % 10))
   => -8"
  {:added "3.0"}
  ([expr & forms]
   `(let [~'% ~expr
          ~@(thread-form forms (fn [form] (concat form ['%])))]
      ~'%)))

(defn require
  ([sym]
   (#'clojure.core/serialized-require sym)))

(defn dev?
  "checks if current environment is dev
 
   (dev?)
   => boolean?"
  {:added "3.0"}
  ([]
   (-> (try
         (the-ns 'nrepl.core)
         (catch Exception e))
       boolean)))

(defn unbound?
  "checks if a variable is unbound
 
   (declare -lost-)
 
   (unbound? -lost-)
   = true"
  {:added "3.0"}
  ([obj]
   (instance? clojure.lang.Var$Unbound obj)))

(defn byte?
  "Returns `true` if `x` is of type `java.lang.Byte`
 
   (byte? (byte 1)) => true"
  {:added "3.0"}
  ([x] (instance? java.lang.Byte x)))

(defn short?
  "Returns `true` if `x` is of type `java.lang.Short`
 
   (short? (short 1)) => true"
  {:added "3.0"}
  [x] (instance? java.lang.Short x))

(defn long?
  "Returns `true` if `x` is of type `java.lang.Long`
 
   (long? 1)          => true
   (long? 1N)         => false"
  {:added "3.0"}
  [x] (instance? java.lang.Long x))

(defn bigint?
  "Returns `true` if `x` is of type `clojure.lang.BigInt`.
 
   (bigint? 1N)       => true
   (bigint? 1)        =>  false"
  {:added "3.0"}
  [x] (instance? clojure.lang.BigInt x))

(defn bigdec?
  "Returns `true` if `x` is of type `java.math.BigDecimal`.
 
   (bigdec? 1M)       => true
   (bigdec? 1.0)      => false"
  {:added "3.0"}
  [x] (instance? java.math.BigDecimal x))

(defn regexp?
  "Returns `true` if `x` implements `clojure.lang.IPersistentMap`.
 
   (regexp? #\"\\d+\") => true"
  {:added "3.0"}
  ([x] (instance? java.util.regex.Pattern x)))

(defn hash-map?
  "Returns `true` if `x` implements `clojure.lang.APersistentMap`.
 
   (hash-map? {})    => true
   (hash-map? [])    => false"
  {:added "3.0"}
  ([x] (instance? clojure.lang.APersistentMap x)))

(defn lazy-seq?
  "Returns `true` if `x` implements `clojure.lang.LazySeq`.
 
   (lazy-seq? (map inc [1 2 3]))  => true
   (lazy-seq? ())    => false"
  {:added "3.0"}
  ([x] (instance? clojure.lang.LazySeq x)))

(defn iobj?
  "checks if a component is instance of clojure.lang.IObj
 
   (iobj? 1) => false
 
   (iobj? {}) => true"
  {:added "3.0"}
  ([x]
   (instance? clojure.lang.IObj x)))

(defn iref?
  "Returns `true` if `x` is of type `clojure.lang.IRef`.
 
   (iref? (atom 0))  => true"
  {:added "3.0"}
  ([obj]
   (instance? clojure.lang.IRef obj)))

(defn ideref?
  "checks if 
 
   (ideref? (volatile! 0)) => true 
   (ideref? (promise)) => true"
  {:added "3.0"}
  ([obj]
   (instance? clojure.lang.IDeref obj)))

(defn promise?
  "Returns `true` is `x` is a promise
 
   (promise? (promise)) => true
   (promise? (future))  => false"
  {:added "3.0"}
  ([^Object obj]
   (let [^String s (.getName ^Class (type obj))]
     (.startsWith s "clojure.core$promise$"))))

(defn thread?
  "Returns `true` is `x` is a thread
 
   (thread? (Thread/currentThread)) => true"
  {:added "3.0"}
  [obj]
  (instance? java.lang.Thread obj))

(defn url?
  "Returns `true` if `x` is of type `java.net.URL`.
 
   (url? (java.net.URL. \"file:/Users/chris/Development\")) => true"
  {:added "3.0"}
  ([x] (instance? java.net.URL x)))

(defn atom?
  "Returns `true` if `x` is of type `clojure.lang.Atom`.
 
   (atom? (atom nil)) => true"
  {:added "3.0"}
  ([obj]
   (instance? clojure.lang.Atom obj)))

(defn comparable?
  "Returns `true` if `x` and `y` both implements `java.lang.Comparable`.
 
   (comparable? 1 1) => true"
  {:added "3.0"}
  ([x y]
   (and (instance? Comparable x)
        (instance? Comparable y)
        (= (type x) (type y)))))

(defn array?
  "checks if object is a primitive array
 
   (array? (into-array []))
   => true
 
   (array? String (into-array [\"a\" \"b\" \"c\"]))
   => true"
  {:added "3.0"}
  ([x]
   (.isArray ^Class (type x)))
  ([cls x]
   (and (array? x)
        (= cls (.getComponentType ^Class (type x))))))

(defn edn?
  "checks if an entry is valid edn
 
   (edn? 1) => true
 
   (edn? {}) => true
 
   (edn? (java.util.Date.))
   => false"
  {:added "3.0"}
  [x]
  (or (nil? x)
      (boolean? x)
      (string? x)
      (char? x)
      (symbol? x)
      (keyword? x)
      (number? x)
      (seq? x)
      (vector? x)
      (record? x)
      (map? x)
      (set? x)
      (tagged-literal? x)
      (var? x)
      (regexp? x)))

(defn aget
  "typesafe aget
 
   (aget (int-array [1 2 3]) 2)
   => 3"
  {:added "3.0"}
  ([arr i]
   (case (.getName ^Class (type arr))
     "[J" (clojure.core/aget ^longs arr i)
     "[I" (clojure.core/aget ^ints arr i)
     "[S" (clojure.core/aget ^shorts arr i)
     "[B" (clojure.core/aget ^bytes arr i)
     "[Z" (clojure.core/aget ^booleans arr i)
     "[F" (clojure.core/aget ^floats arr i)
     "[D" (clojure.core/aget ^doubles arr i))))

(defn seqify
  "if not a sequence, then make one
 
   (seqify 1)
   => [1]
   
   (seqify [1])
   => [1]"
  {:added "3.0"}
  ([x]
   (if (coll? x)
     x
     [x])))

(defn unseqify
  "if a sequence, takes first element
 
   (unseqify [1])
   => 1
   
   (unseqify 1)
   => 1"
  {:added "3.0"}
  ([x]
   (if (coll? x)
     (first x)
     x)))

(defn map-keys
  "changes the keys of a map
 
   (map-keys inc {0 :a 1 :b 2 :c})
   => {1 :a, 2 :b, 3 :c}"
  {:added "3.0"}
  [f m]
  (reduce (fn [out [k v]]
            (assoc out (f k) v))
          {}
          m))

(defn map-vals
  "changes the values of a map
 
   (map-vals inc {:a 1 :b 2 :c 3})
   => {:a 2, :b 3, :c 4}"
  {:added "3.0"}
  [f m]
  (reduce (fn [out [k v]]
            (assoc out k (f v)))
          {}
          m))

(defn map-entries
  "manipulates a map given the function
 
   (map-entries (fn [[k v]]
                  [(keyword (str v)) (name k)])
                {:a 1 :b 2 :c 3})
   => {:1 \"a\", :2 \"b\", :3 \"c\"}"
  {:added "3.0"}
  [f m]
  (->> (map f m)
       (into {})))

(defn filter-keys
  "filters map based upon map keys
   
   (filter-keys even? {0 :a 1 :b 2 :c})
   => {0 :a, 2 :c}"
  {:added "3.0"}
  ([pred m]
   (reduce (fn [out [k v]]
             (if (pred k)
               (assoc out k v)
               out))
           {}
           m)))

(defn filter-vals
  "filters map based upon map values
 
   (filter-vals even? {:a 1 :b 2 :c 3})
   => {:b 2}"
  {:added "3.0"}
  [pred m]
  (reduce (fn [out [k v]]
            (if (pred v)
              (assoc out k v)
              out))
          {}
          m))

(defn qualified-keys
  "takes only the namespaced keys of a map
 
   (qualified-keys {:a 1 :ns/b 1})
   => #:ns{:b 1}
 
   (qualified-keys {:a 1 :ns.1/b 1 :ns.2/c 1}
                   :ns.1)
   => #:ns.1{:b 1}"
  {:added "3.0"}
  ([m]
   (filter-keys (fn [k] (and (keyword? k)
                             (namespace k)))
                m))
  ([m ns]
   (let [ns (keystring ns)]
     (filter-keys (fn [k] (and (keyword? k)
                               (= (namespace k) ns)))
                  m))))

(defn unqualified-keys
  "takes only the namespaced keys of a map
   
   (unqualified-keys {:a 1 :ns/b 1})
   => {:a 1}"
  {:added "3.0"}
  ([m]
   (filter-keys (fn [k] (and (keyword? k)
                             (nil? (namespace k))))
                m)))

(defn qualify-keys
  "lifts all unqualified keys
 
   (qualify-keys {:a 1} :ns.1)
   => #:ns.1{:a 1}
 
   (qualify-keys {:a 1 :ns.2/c 1} :ns.1)
   => {:ns.1/a 1, :ns.2/c 1}"
  {:added "3.0"}
  ([m ns]
   (let [ns (keystring ns)]
     (map-keys (fn [k]
                 (if (and (keyword? k)
                          (not (namespace k)))
                   (clojure.core/keyword ns (name k))
                   k))
               m))))

(defn unqualify-keys
  "unqualifies keys in the map
 
   (unqualify-keys {:a 1 :ns.1/b 1 :ns.2/c 1})
   => {:a 1, :b 1, :c 1}
 
   (unqualify-keys {:a 1 :ns.1/b 1 :ns.2/c 1} :ns.1)
   => {:a 1, :b 1, :ns.2/c 1}"
  {:added "3.0"}
  ([m]
   (map-keys (fn [k] (clojure.core/keyword (name k))) m))
  ([m ns]
   (let [ns (keystring ns)]
     (map-keys (fn [k]
                 (if (and (keyword? k)
                          (= (namespace k) ns))
                   (clojure.core/keyword (name k))
                   k))
               m))))
               
(defn cursor
  "adds a cursor to the atom to update on any change
 
   (def a (atom {:a {:b 1}}))
   
   (def ca (cursor a [:a :b]))
 
   (do (swap! ca + 10)
       (swap! a update-in [:a :b] + 100)
       [(deref a) (deref ca)])
   => [{:a {:b 111}} 111]"
  {:added "3.0"}
  ([ref selector]
   (cursor ref selector (str (uuid))))
  ([ref selector key]
   (let [getter  (fn [m] (get-in m selector))
         setter  (fn [m v] (assoc-in m selector v))
         initial (getter @ref)
         cursor  (atom initial)]
     (add-watch ref key (fn [_ _ _ v]
                          (let [cv (getter v)]
                            (if (not= cv @cursor)
                              (clojure.core/reset! cursor cv)))))
     (add-watch cursor key (fn [_ _ _ v]
                             (clojure.core/swap! ref setter v)))
     cursor)))

(defn derived
  "constructs an atom derived from other atoms
 
   (def a (atom 1))
   (def b (atom 10))
   (def c (derived [a b] +))
 
   (do (swap! a + 1)
       (swap! b + 10)
       [@a @b @c])
   => [2 20 22]"
  {:added "3.0"}
  ([atoms f]
   (derived atoms f (str (uuid))))
  ([atoms f key]
   (let [cache     (volatile! (map deref atoms))
         derived-fn #(apply f @cache)
         derived  (atom (derived-fn))]
     (doseq [atom atoms]
       (add-watch atom key
                  (fn [_ _ _ _]
                    (let [results (map deref atoms)]
                      (when (not= @cache results)
                        (vreset! cache results)
                        (clojure.core/reset! derived (derived-fn)))))))
     derived)))

(defn unfold
  "unfolds using a generated function
 
   (unfold (fn [[i :as seed]]
             (if i
               (if-not (neg? i)
                 [(* i 2) [(dec i)]])))
           [10])
   => [20 18 16 14 12 10 8 6 4 2 0]"
  {:added "3.0"}
  ([f coll]
   (->> coll
        (list nil)
        (iterate (comp f second))
        rest
        (take-while some?)
        (map first))))

(defn swap-return!
  "returns output and new state of atom
 
   (swap-return! (atom {})
                 (fn [m]
                   [true (assoc m :a 1)]))
   => [true {:a 1}]"
  {:added "3.0"}
  ([atm f & args]
   (let [output (volatile! nil)
         new (swap! atm (fn [v]
                          (let [[return new] (apply f v args)
                                _ (vreset! output return)]
                            new)))]
     [@output new])))

(defn demunge
  "Given a string representation of a fn class,
  as in a stack trace element, returns a readable version."
  {:added "1.3"}
  ([fn-name]
   (clojure.lang.Compiler/demunge fn-name)))

(defn say
  "enables audio debugging
 
   (say \"hello there\")"
  {:added "3.0"}
  ([& phrase]
   (future (apply sh/sh "say" (map str phrase)))))

(defn beep
  ([]
   (if (or (native?) *native-compile*)
     (sh/sh "/bin/bash" "-c" "echo -ne '\\007' > /dev/tty")
     (eval '(.beep (java.awt.Toolkit/getDefaultToolkit))))))

(defn wait-for-port
  ([{:keys [host port default timeout pause] :as opts}]
   (let [t0 (time-ns)]
     (loop [retries 0]
       (let [t1 (try (.close (Socket. ^String host ^long port))
                     (time-ns)
                     (catch ConnectException _
                       (let [t1 (time-ns)]
                         (if (and timeout
                                  (>= (- t1 t0) timeout))
                           :port/timed-out
                           :port/try-again))))]
         (cond (identical? :port/retry t1)
               (do (Thread/sleep (or pause 100))
                   (recur (inc retries)))

               (identical? :port/timed-out t1)
               (throw (ex-info "Timed out" {:host :host :port port}))
               
               :else
               {:host host :port port :start t0 :end t1 :retries retries}))))))


(comment
  (bench-ns (.beep (java.awt.Toolkit/getDefaultToolkit)))
  262541
  (bench-ns (eval '(.beep (java.awt.Toolkit/getDefaultToolkit))))
  800405
  
  )
