(ns hara.code.unit
  (:require [clojure.set :as set]
            [hara.string :as string]
            [hara.code.framework :as base]
            [hara.code.framework.docstring :as docstring]
            [hara.code.unit.scaffold :as scaffold]
            [hara.code.unit.walk :as walk]
            [hara.core.base.result :as result]
            [hara.data.base.map :as map]
            [hara.data.base.seq.diff :as seq.diff]
            [hara.lib.diff :as diff]
            [hara.io.file :as fs]
            [hara.io.project :as project]
            [hara.function.task :as task]
            [hara.code.block :as block])
  (:refer-clojure :exclude [import]))

(defn import
  "imports unit tests as docstrings
 
   (project/in-context (import {:print {:function true}}))
   => map?"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns   (project/source-ns ns)
         test-ns     (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)
         params  (task/single-function-print params)]
     (cond (nil? source-file)
           (result/result {:status :error
                           :data :no-source-file})

           (nil? test-file)
           (result/result {:status :error
                           :data :no-test-file})

           :else
           (let [import-fn    (fn [nsp refers]
                                (fn [zloc]
                                  (docstring/insert-docstring zloc nsp refers)))
                 refers       (base/analyse test-ns params lookup project)
                 transform-fn (fn [text] (walk/walk-string text '_ refers import-fn))]
             (base/transform-code source-ns (assoc params :transform transform-fn) lookup project))))))

(defn purge
  "purge docstrings and meta from file
 
   (project/in-context (purge {:print {:function true}}))
   => map?"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns  (project/source-ns ns)
         source-file (lookup source-ns)
         params  (task/single-function-print params)]
     (cond (nil? source-file)
           (result/result {:status :info
                           :data :no-source-file})

           :else
           (let [purge-fn (fn [nsp references] identity)
                 transform-fn (fn [text] (walk/walk-string text '_ {} purge-fn))]
             (base/transform-code source-ns (assoc params :transform transform-fn) lookup project))))))

(defn missing
  "returns all functions missing unit tests
 
   (project/in-context (missing))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns (project/source-ns ns)
         test-ns (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)]
     (cond (nil? source-file)
           (result/result {:status :error
                           :data :no-source-file})

           :else
           (let [source-vars (if source-file (base/vars source-ns params lookup project))
                 test-vars   (if test-file   (base/vars test-ns params lookup project))]
             (vec (sort (set/difference (set source-vars) (set test-vars)))))))))

(defn todos
  "returns all unit tests with TODOs
 
   (project/in-context (todos))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [test-ns   (project/test-ns ns)
         test-file (lookup test-ns)]
     (cond (nil? test-file)
           (result/result {:status :error
                           :data :no-test-file})

           :else
           (->> (read-string (str "[" (slurp test-file) "]"))
                (filter (fn [form]
                          (and (list? form)
                               (->> form (take 2) (= '(fact "TODO"))))))
                (mapv (comp symbol name :refer meta)))))))

(defn incomplete
  "returns functions with todos all missing tests
 
   (project/in-context (incomplete))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns (project/source-ns ns)
         test-ns   (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)]
     (-> (concat
          (if source-file (missing source-ns params lookup project))
          (if test-file (todos test-ns params lookup project)))
         sort
         vec))))

(defn orphaned-meta
  "returns true if meta satisfies the orphaned criteria
 
   (orphaned-meta {}) => true
 
   (orphaned-meta {:meta {:adopt true} :ns 'clojure.core :var 'slurp})
   => false
 
   (orphaned-meta {:meta {:adopt true} :ns 'clojure.core :var 'NONE})
   => true"
  {:added "3.0"}
  [m]
  (not (and (-> m :meta :adopt)
            (try
              (require (:ns m))
              (resolve (symbol (str (:ns m))
                               (first (string/split (str (:var m)) #"\."))))
              (catch Exception e)))))

(defn orphaned
  "returns unit tests that do not have an associated function
 
   (project/in-context (orphaned))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns (project/source-ns ns)
         test-ns (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)
         params (map/merge-new params {:full true})]
     (cond (nil? test-file)
           (result/result {:status :error
                           :data :no-test-file})

           :else
           (let [source-vars (if source-file (base/vars source-ns params lookup project))
                 test-vars   (if test-file   (base/vars test-ns params lookup project))
                 orphaned    (set/difference (set test-vars) (set source-vars))
                 orphaned    (filter (comp orphaned-meta meta) orphaned)]
             (mapv (fn [sym]
                     (if (= (namespace sym) (str source-ns))
                       (symbol (name sym))
                       sym))
                   (sort orphaned)))))))

(defn mark-vars
  "captures changed vars in a set
 
   (mark-vars '[a1 a2 a3 a4 a5]
              '[a1 a4 a3 a2 a5])
   => '[2 [a1 #{a2} #{a3} a4 a5]]"
  {:added "3.0"}
  [vars comp-vars]
  (let [marked-set (->> (second (seq.diff/diff vars comp-vars))
                        (filter (comp #(= :+ %) first))
                        (mapcat #(nth % 2))
                        set)
        results (mapv (fn [var]
                        (let [marked? (marked-set var)
                              sym (symbol (name var))]
                          (if marked?
                            (set [sym])
                            sym)))
                      vars)]
    [(count (filter set? results)) results]))

(defn in-order?
  "determines if the test code is in the same order as the source code
 
   (project/in-context (in-order?))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [source-ns (project/source-ns ns)
         test-ns   (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)
         params (map/merge-new params {:full true})]
     (cond (nil? (or test-file source-file))
           (result/result {:status :error
                           :data :invalid-namespace})

           (nil? test-file)
           (result/result {:status :error
                           :data :no-test-file})

           (nil? source-file)
           (result/result {:status :error
                           :data :no-source-file})

           :else
           (let [source-vars (base/vars source-ns params lookup project)
                 test-vars   (base/vars test-ns params lookup project)
                 orphaned    (set/difference (set test-vars) (set source-vars))
                 test-vars   (vec (remove orphaned test-vars))]
             (cond (= source-vars test-vars)
                   []

                   :else
                   (let [[count source-marked] (mark-vars source-vars test-vars)]
                     (if (pos? count)
                       [count source-marked (second (mark-vars test-vars source-vars))]
                       []))))))))

(defn scaffold
  "creates a set of tests for a given source
 
   (project/in-context (scaffold))"
  {:added "3.0"}
  ([ns {:keys [write print] :as params} lookup project]
   (let [source-ns (project/source-ns ns)
         test-ns   (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)
         version (->> (string/split (:version project) #"\.")
                      (take 2)
                      (string/join "."))
         source-vars (if source-file (base/vars source-ns {:sorted false} lookup project))
         test-vars   (if test-file   (base/vars test-ns {:sorted false} lookup project))
         new-vars    (seq (remove (set test-vars) source-vars))
         params  (task/single-function-print params)]
     (cond (nil? source-file)
           (result/result {:status :error
                           :data :no-source-file})

           (empty? source-vars)
           (result/result {:status :info
                           :data :no-source-vars})

           :else
           (let [transform-fn (if test-file
                                (fn [original] (scaffold/scaffold-append original source-ns new-vars version))
                                (fn [_] (scaffold/scaffold-new source-ns test-ns source-vars version)))
                 [original test-file]  (if test-file
                                         [(slurp test-file) test-file]
                                         ["" (scaffold/new-filename test-ns project write)])
                 params (assoc params :transform transform-fn)
                 result (base/transform-code test-ns params (assoc lookup test-ns test-file) project)]
             (assoc result :new (vec new-vars)))))))

(defn arrange
  "arranges the test code to be in the same order as the source code
 
   (project/in-context (arrange))"
  {:added "3.0"}
  ([ns params lookup project]
   (let [ret (in-order? ns params lookup project)
         source-ns   (project/source-ns ns)
         test-ns     (project/test-ns ns)
         source-file (lookup source-ns)
         test-file   (lookup test-ns)
         params  (task/single-function-print params)]
     (cond (result/result? ret)
           ret

           (empty? ret)
           {:changes [] :updated false :path test-file}

           (nil? source-file)
           (result/result {:status :error
                           :data :no-source-file})

           (nil? test-file)
           (result/result {:status :error
                           :data :no-test-file})
           :else
           (let [source-vars  (base/vars source-ns {:sorted false} lookup project)
                 transform-fn (fn [original]
                                (scaffold/scaffold-arrange original source-vars))]
             (base/transform-code test-ns (assoc params :transform transform-fn) lookup project))))))

(defn unchecked
  "returns tests that does not contain a `=>`
 
   (project/in-context (unchecked))"
  {:added "3.0"}
  [ns params lookup project]
  (let [source-ns (project/source-ns ns)
        test-ns   (project/test-ns ns)
        test-file (lookup test-ns)
        analysis  (base/analyse test-ns params lookup project)]
    (cond (result/result? analysis)
          analysis

          :else
          (let [entries (get analysis source-ns)]
            (vec (keep (fn [[var entry]]
                         (if (and (->> (get-in entry [:test :form])
                                       (not= 'comment))
                                  (->> (get-in entry [:test :code])
                                       (map block/string)
                                       (filter #(= "=>" %))
                                       (empty?)))
                           var))
                       entries))))))

(defn commented
  "returns tests that are in a comment block
 
   (project/in-context (commented))"
  {:added "3.0"}
  [ns params lookup project]
  (let [source-ns (project/source-ns ns)
        test-ns   (project/test-ns ns)
        analysis  (base/analyse test-ns params lookup project)]
    (cond (result/result? analysis)
          analysis

          :else
          (let [entries (get analysis source-ns)]
            (vec (keep (fn [[var entry]]
                         (if (->> (get-in entry [:test :form])
                                  (= 'comment))
                           var))
                       entries))))))

(defn pedantic
  "returns all probable improvements on tests
 
   (project/in-context (pedantic))"
  {:added "3.0"}
  [ns params lookup project]
  (let [source-ns (project/source-ns ns)
        tag (fn [k inputs]
              (if-not (result/result? inputs)
                (map #(vector % k) inputs)))]
    (-> (concat (tag :M (missing ns params lookup project))
                (tag :T (todos ns params lookup project))
                (tag :N (unchecked ns params lookup project))
                (tag :C (commented ns params lookup project)))
        sort
        vec)))
