(ns hara.deploy.maven.command
  (:require [hara.core.base.encode :as encode]
            [hara.deploy.maven.package :as package]
            [hara.io.archive :as archive]
            [hara.io.file :as fs]
            [hara.lib.aether :as aether]
            [hara.security :as security]
            [hara.security.openpgp :as openpgp]))

(def +home+        (System/getProperty "user.home"))
(def +key-dir+     (str +home+ "/.hara/keys"))
(def +config+      (str +home+ "/.hara/maven.edn"))

(defn read-config
  ([] (read-config +config+ +key-dir+))
  ([config key-dir]
   (let [{:keys [profile repositories]} (read-string (slurp config))
         {:keys [user key deploy]} profile
         secret-key (openpgp/parse-secret-key
                     (slurp (str +key-dir+ "/" user ".sec.gpg")))
         [public-key private-key] (openpgp/key-pair secret-key)]
     {:suffix "asc"
      :public-key public-key
      :private-key private-key
      :repository (get repositories deploy)})))

(def +default+ (read-config +config+ +key-dir+))

(defn sign-file
  "signs a file with gpg
 
   (sign-file {:file \"project.clj\" :extension \"clj\"}
              {:signing (-> hara.security.gpg.local/LEIN-PROFILE
                           slurp
                            read-string
                            :user
                            :signing
                            :gpg-key)})"
  {:added "1.2"}
  ([{:keys [file extension]}
    {:keys [suffix public-key private-key] :as params}]
   (let [output (str file "." suffix)
         output-ex (str extension "." suffix)]
     (openpgp/sign file output params)
     {:file output
      :extension output-ex})))

(defn create-digest
  "creates a digest given a file and a digest type
 
   (create-digest \"MD5\"
                  \"md5\"
                  {:file \"project.clj\"
                   :extension \"clj\"})
   => {:file \"project.clj.md5\",
       :extension \"clj.md5\"}"
  {:added "1.2"}
  [algorithm suffix {:keys [file extension] :as artifact}]
  (let [content (-> (fs/read-all-bytes file)
                    (security/digest "MD5")
                    (encode/to-hex))
        file (str file "." suffix)
        extension (str extension "." suffix)
        _ (spit file content)]
    {:file file :extension extension}))

(defn add-digest
  "adds MD5 and SHA1 digests to all artifacts
 
   (add-digest [{:file \"project.clj\",
                 :extension \"clj\"}])
   => [{:file \"project.clj.md5\", :extension \"clj.md5\"}
       {:file \"project.clj\", :extension \"clj\"}]"
  {:added "1.2"}
  [artifacts]
  (concat (mapv (partial create-digest "MD5" "md5") artifacts)
          artifacts))

(defn clean
  "cleans the interim directory
 
   (clean 'hara.function.task
          {}
          {}
          (project/project))"
  {:added "3.0"}
  [name {:keys [simulate interim] :as params} linkages project]
  (let [interim (or interim package/+interim+)
        output (fs/path (:root project) interim (str name))]
    (fs/delete output {:simulate simulate})))

(defn install
  "installs a package to the local `.m2` repository
 
   (install 'hara.function.task
            {}
            {}
            (assoc (project/project)
                   :plan (analyser/create-plan (project/project))
                   :aether (aether/aether)))"
  {:added "3.0"}
  [name {:keys [skip bulk] :as params} linkages {:keys [aether] :as project}]
  (let [process (if skip package/infer package/package)
        {:keys [jar pom interim] :as rep} (process name params linkages project)]
    (aether/install-artifact aether
                             rep
                             {:artifacts [{:file (fs/path interim jar)
                                           :extension "jar"}
                                          {:file (fs/path interim pom)
                                           :extension "pom"}]
                              :print {:title (not bulk)
                                      :timing (not bulk)}})))

(defn install-secure
  "installs a package to the local `.m2` repository with signing
 
   (install-secure 'hara.function.task
                   {}
                   {}
                   (assoc (project/project)
                          :plan (analyser/create-plan (project/project))
                          :aether (aether/aether)))"
  {:added "3.0"}
  [name {:keys [skip bulk suffix public-key private-key] :as params} linkages {:keys [aether] :as project}]
  (let [process (if skip package/infer package/package)
        {:keys [jar pom interim] :as rep} (process name params linkages project)
        artifacts  [{:file (fs/path interim jar)
                     :extension "jar"}
                    {:file (fs/path interim pom)
                     :extension "pom"}]
        artifacts (->> artifacts
                       (map #(sign-file % params))
                       (concat artifacts)
                       (add-digest))]
    (aether/install-artifact aether
                             rep
                             {:artifacts artifacts
                              :print {:title (not bulk)
                                      :timing (not bulk)}})))

(defn deploy
  "deploys a package 
 
   (deploy 'hara.function.task
           {:repository {:id \"caudata\"
                         :url \"https://maven.caudata.io\"
                         :authentication {:username \"caudata\"
                                          :password \"caudata\"}}}
           {}
           (assoc (project/project)
                  :plan (analyser/create-plan (project/project))
                  :aether (aether/aether)))"
  {:added "3.0"}
  ([name {:keys [skip bulk repository] :as params} linkages {:keys [aether] :as project}]
   (let [{:keys [url authentication digest] :or {digest true}} repository
         process (if skip package/infer package/package)
         {:keys [jar pom interim] :as rep}  (process name params linkages project)
         artifacts  [{:file (fs/path interim jar)
                      :extension "jar"}
                     {:file (fs/path interim pom)
                      :extension "pom"}]
         artifacts (->> artifacts
                       (map #(sign-file % params))
                       (concat artifacts))
         artifacts (if digest (add-digest artifacts) artifacts)]
     (aether/deploy-artifact aether
                             rep
                             {:artifacts artifacts
                              :repository repository
                              :print {:title  (not bulk)
                                      :timing (not bulk)}}))))
