(ns hara.io.concurrent
  (:import (java.util.concurrent Executors
                                 ExecutorService
                                 ThreadPoolExecutor
                                 TimeUnit
                                 BlockingQueue
                                 BlockingDeque
                                 ArrayBlockingQueue
                                 LinkedBlockingQueue
                                 LinkedBlockingDeque
                                 SynchronousQueue))
  (:refer-clojure :exclude [take pop]))

(defn ^LinkedBlockingDeque deque
  []
  (LinkedBlockingDeque.))

(defn ^BlockingQueue queue
  []
  (LinkedBlockingQueue.))

(defn ^BlockingQueue fixed-queue
  [size]
  (ArrayBlockingQueue. size))

(defn ^ExecutorService single-executor
  ([]
   (Executors/newSingleThreadExecutor))
  ([size]
   (ThreadPoolExecutor. 0 1 0 TimeUnit/MILLISECONDS (fixed-queue size))))

(defn shutdown
  ([^ExecutorService service]
   (.shutdown service)))

(defn shutdown-now
  ([^ExecutorService service]
   (.shutdownNow service)))

(defn ^ExecutorService pool-executor
  ([size max keep-alive]
   (pool-executor size max keep-alive (queue)))
  ([^long size ^long max ^long keep-alive ^BlockingQueue queue]
   (ThreadPoolExecutor. size max keep-alive TimeUnit/MILLISECONDS queue)))

(defn ^BlockingQueue get-queue
  ([service]
   (if (= (type service) ThreadPoolExecutor)
     (.getQueue ^ThreadPoolExecutor service)
     (throw (ex-info "Cannot access queue" {})))))

(defn ^Callable wrap-min-time
  ([f ms]
   (fn []
     (let [start (System/currentTimeMillis)
           _ (f)
           end (System/currentTimeMillis)
           duration (- end start)]
       (if (> ms duration)
         (Thread/sleep (- ms duration)))))))

(defn submit
  ([^ExecutorService service ^Callable f]
   (submit service f nil))
  ([^ExecutorService service f min-time] 
   (let [^Callable f (cond-> f
                       min-time (wrap-min-time min-time))]
     (.submit service f))))

(defn submit-notify
  ([^ExecutorService service ^Callable f]
   (submit-notify service f nil))
  ([^ExecutorService service f min-time]
   (if (pos? (.remainingCapacity (get-queue service)))
     (try (submit service f min-time)
          (catch Throwable t)))))

(defn take
  ([^BlockingQueue queue]
   (.take queue)))

(defn add
  ([^BlockingQueue queue element]
   (.add queue element)))

(defn push
  ([^BlockingDeque queue element]
   (.push queue element)))

(defn pop
  ([^BlockingDeque queue]
   (.pop queue)))

(defn process-queue
  [f queue maximum]
  (loop [total   (count queue)]
   (let [n      (if (> total maximum)
                  maximum
                total)
         txs  (->> (repeatedly n #(take queue))
                   vec)
         _   (when-not (empty? txs)
               (f txs))]
     (if (= n maximum)
       (recur (count queue))))))
