(ns hara.net.http.handler
  (:require [hara.io.file :as fs]
            [hara.event :as event]
            [hara.core.base.result :as result]
            [hara.net.http.common :as common]))

(declare match-handler)

(defn match-route
  "matches route for handler and request
 
   (match-route {:method :get
                 :route \"hello\"}
                {:method :get
                 :route \"hello\"})
   => true"
  {:added "3.0"}
  [{:keys [method route] :as opts} req]
  ;; TODO - make extensible and return route params
  (and (or (nil? method)
           (= method (:method req)))
       (or (nil? route)
           (= route (:route req)))))

(defn trim-seperators
  "gets rid of seperators on ends of string
 
   (trim-seperators \"//api/me//\")
   => \"api/me\""
  {:added "0.5"}
  [s]
  (let [ns (cond-> s
             (.startsWith s "/")
             (-> (subs 1))
             
             (.endsWith s "/")
             (-> (subs 0 (dec (count s)))))]
    (if (= (count ns) (count s))
      ns
      (trim-seperators ns))))

(defrecord Handler [type route match]
  
  Object
  (toString [handler]
    (str "#handler " [type route (into {} (dissoc handler :type :route :match))]))
  
  clojure.lang.IFn
  (invoke [handler req]
    (if (match req)
      ((:fn handler) req))))

(defmethod print-method Handler
  [v w]
  (.write w (str v)))

(defn any-handler
  "invokes function when request matches route
 
   ((any-handler {:route \"/hello\"
                  :fn (fn [req] {:status :return})})
    {:route \"/hello\"})
   => {:status :return}"
  {:added "3.0"}
  [{:keys [route] :as opts}]
  (let [match-fn (partial match-route (dissoc opts :method))]
    (map->Handler (assoc opts :match match-fn))))

(defn method-handler
  "invokes function when request matches route and method
 
   ((method-handler {:method :get
                     :route \"/hello\"
                     :fn (fn [req] {:status :return})})
    {:method :get
     :route \"/hello\"})
   => {:status :return}"
  {:added "3.0"}
  [{:keys [type] :as opts}]
  (let [match-fn (partial match-route opts)]
    (map->Handler (assoc opts :match match-fn))))

(defn resource-handler
  "returns a file on the resource path
 
   ((resource-handler {:route \"/resource\"})
    {:method :get
     :route \"/resource/project.clj\"})
   => (contains {:status 200})"
  {:added "3.0"}
  [{:keys [route root] :as opts}]
  (let [match-fn   (fn [{:keys [method] :as req}]
                     (and (= method :get)
                          (.startsWith (:route req)
                                       route)))
        handler-fn (fn [req]
                     (let [path (-> (subs (:route req)
                                          (count route))
                                    (trim-seperators))
                           root (if root (str root "/") "")
                           url  (fs/resource (str root path))]
                       (if url
                         {:status 200
                          :body (.openStream url)})))]
    (map->Handler (assoc opts
                         :fn handler-fn
                         :match match-fn))))  

(defn endpoint-handler-apply
  "applies arguments to the handler
 
   (endpoint-handler-apply {:add (fn [a b] (+ a b))}
                           :add
                           [1 2])
   => (contains {:data 3, :id :add})"
  {:added "3.0"}
  [functions id args]
  (let [func (get functions id)
        output (cond (nil? func)
                     (result/result {:type :http
                                     :status :error
                                     :message "No function found"
                                     :data {:options (keys functions)}})
                     :else
                     (event/manage
                      (let [result (apply func args)]
                        (result/result {:type :http
                                        :status :return
                                        :data result}))
                      (event/on _ issue
                                (result/result {:type    :http
                                                :status  :error
                                                :data    issue}))
                      (catch clojure.lang.ExceptionInfo e
                        (result/result {:type :http
                                        :status  :error
                                        :message (ex-info e)
                                        :data    (ex-data e)}))
                      (catch Throwable t
                        (result/result {:type :http
                                        :status  :error
                                        :message (.getMessage t)}))))]
    (assoc output :id id)))

(defn endpoint-handler
  "constructs a endpoint handler
 
   (-> ((endpoint-handler {:route \"/app\"
                           :functions {:add (fn [a b] (+ a b))}})
        {:route \"/app\"
         :body (str {:id :add
                     :args [1 2]})})
       :body
       read-string)
   => (contains {:id :add, :status :return, :data 3})"
  {:added "3.0"}
  [{:keys [route
           functions
           format] :as opts
    :or {format :edn}}]
  (let [match-fn (partial match-route (dissoc opts :method))
        handler-fn (fn [{:keys [body] :as req}]
                     (let [body (try
                                  (common/-read-body body format)
                                  (catch Throwable t
                                    (result/result {:type    :http
                                                    :status  :error
                                                    :message "Unable to read body"
                                                    :data {:body body :format format}})))
                           
                           output (if (result/result? body)
                                    body
                                    (let [{:keys [id type args]
                                           :or {type :http}} body]
                                      (endpoint-handler-apply functions id args)))]
                       {:status 200
                        :body (common/-write-value (into {} output) format)}))]
    (map->Handler (assoc opts
                         :fn handler-fn
                         :match match-fn))))

(defrecord MultiHandler [routes]
  clojure.lang.IFn
  (invoke [_ req]
    (some #(% req) routes)))

(defn multi-handler
  "chains multiple handlers together
 
   ((multi-handler [(method-handler {:method :get
                                     :fn (constantly {:status :get})})
                    (any-handler    {:fn (constantly {:status :any})})])
    {:method :post})
   => {:status :any}"
  {:added "3.0"}
  [routes]
  (->MultiHandler routes))
