(ns hara.platform.cache
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.core.component :as component]
            [hara.platform.cache.base :deps true])
  (:refer-clojure :exclude [get set keys count]))

(defn set
  "sets the value with or optional expiry
 
   (-> (cache)
       (cache/set :a 1)
       :state
       deref)
   => {:a {:value 1}}"
  {:added "3.0"}
  ([cache key value]
   (protocol.cache/-set cache key value))
  ([cache key value expiry]
   (protocol.cache/-set cache key value expiry)))

(defn get
  "returns the value of a key
   
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2} :c {:value 3}}})
       (cache/get :a))
   => 1"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-get cache key))

(defn count
  "returns number of active keys in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/count))
   => 2"
  {:added "3.0"}
  [cache]
  (protocol.cache/-count cache))

(defn batch
  "performs multiple operations in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/batch {:c 3 :d 4} {} [:b])
       (cache/all))
   => {:a 1, :c 3, :d 4}"
  {:added "3.0"}
  ([cache add-values]
   (protocol.cache/-batch cache add-values {} []))
  ([cache add-values add-expiry remove-vec]
   (protocol.cache/-batch cache add-values add-expiry remove-vec)))

(defn delete
  "deletes given key
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/delete :b)
       (cache/all))
   => {:a 1}"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-delete cache key))

(defn clear
  "clears the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/clear)
       (cache/all))
   => {}"
  {:added "3.0"}
  [cache]
  (protocol.cache/-clear cache))

(defn all
  "returns all values in the cache
   
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/all))
   => {:a 1, :b 2}"
  {:added "3.0"}
  [cache]
  (protocol.cache/-all cache))

(defn keys
  "returns all keys in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/keys)
       sort)
   => [:a :b]"
  {:added "3.0"}
  [cache]
  (protocol.cache/-keys cache))

(defn touch
  "renews the expiration time for a given key
   
   (binding [base/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 1001}}})
         (cache/touch :a 10)
         :state
         deref))
   => {:a {:value 1, :expiration 11000}}"
  {:added "3.0"}
  [cache key expiry]
  (protocol.cache/-touch cache key expiry))

(defn expired?
  "checks if a given key time is expired
   
   (binding [base/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 1001}}})
         (cache/expired? :a)))
   => false"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-expired? cache key))

(defn expiry
  "return the expiry of a key in seconds
 
   (binding [base/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 7000}}})
         (cache/expiry :a)))
   => 6"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-expiry cache key))

(defn create
  "creates a cache that is component compatible
 
   (cache/create {:type :mock
                  :file {:path \"dev/scratch/test.edn\"
                         :reset true}})
   ;;=> #cache.mock<uninitiased>
   "
  {:added "3.0"}
  [m]
  (protocol.cache/-create m))

(defn cache
  "creates an active cache
 
   (cache {:type :mock
           :initial {:a {:value 1}}
           :file {:path \"dev/scratch/test.edn\"
                  ;;:load true
                  :reset true}})
   ;;=> #cache.mock{:a 1}
   "
  {:added "3.0"}
  ([] (cache {:type :mock}))
  ([m]
   (-> (create m)
       (component/start))))
