(ns hara.platform.mail.mock
  (:require [hara.protocol.mail :as protocol.mail]
            [hara.protocol.component :as protocol.component]
            [clojure.set :as set]
            [hara.core.component :as component]
            [hara.object :as object]
            [hara.data.table :as table]
            [hara.platform.mail.common :as common]
            [hara.platform.mail.interop.message :as message])
  (:refer-clojure :exclude [send])
  (:import java.util.UUID))

(defonce ^:dynamic *active* (atom {}))

(defn activate
  "puts a given mailbox into the *active* record
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (activate))
 
   (:a @*active*)
   => hara.platform.mail.mock.MockMailbox"
  {:added "3.0"}
  ([mailbox]
   (let [id (or (:id mailbox)
                (str (UUID/randomUUID)))]
     (activate id mailbox)))
  ([id mailbox]
   (if-let [old (get @*active* id)]
     (component/stop old))
   (swap! *active* assoc id mailbox)
   (assoc mailbox :id id)))

(defn deactivate
  "removes a given mailbox into the *active* record
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (activate)
       (deactivate))
 
   (:a @*active*) => nil"
  {:added "3.0"}
  [mailbox]
  (swap! *active* dissoc (:id mailbox))
  (dissoc mailbox :id))

(defrecord MockMailbox [id state addresses]
  Object
  (toString [_]
    (str "#mailbox.mock" (if (and state
                                  (get @*active* id))
                           {:id id
                            :messages (count @state)
                            :addresses  addresses}
                           "<uninitiased>")))

  protocol.mail/IMailbox
  (-init-mail [_ messages]
    (reset! state (mapv common/to-string messages)))

  (-count-mail [_]
    (count @state))

  (-get-mail [_ i]
    (common/from-string (nth @state i)))

  (-list-mail [_]
    (map common/from-string @state))

  (-clear-mail [_]
    (reset! state []))

  protocol.component/IComponent
  (-start [mailbox]
    (-> mailbox table/attach-state activate))

  (-stop [mailbox]
    (-> mailbox deactivate table/detach-state)))

(defmethod print-method MockMailbox
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.mail/-create-mailbox :mock
  [m]
  (-> (assoc m :initial [])
      (update-in [:addresses] set)
      (map->MockMailbox)))

(defn send
  "sends a message to an active mailbox
 
   (-> (mail/create-mailbox {:type :mock
                             :id :a
                             :addresses #{\"a@a.com\"}})
       (component/start))
 
   (count @(:state (:a @*active*)))
   => 0
   
   (send {:to  [\"a@a.com\"]
          :subject \"Test\"
          :body \"Hello There\"})
   
   (count @(:state (:a @*active*)))
   => 1"
  {:added "3.0"}
  [message]
  (let [m   (message/message message)
        recipients (set (object/to-data (object/get m :all-recipients)))
        m   (doto m
              (.removeHeader "Bcc"))]

    (doseq [[_ mailbox] @*active*]
      (let [matching (count (set/intersection recipients (:addresses mailbox)))]
        (if (pos? matching)
          (swap! (:state mailbox) conj (common/to-string m)))))))

(defrecord MockMailer []
  Object
  (toString [_]
    (str "#mailer.mock" {}))

  protocol.mail/IMailer
  (protocol.mail/-send-mail [_ message]
    (send message))

  (protocol.mail/-send-bulk [_ messages]
    (doseq [m messages]
      (send m))))

(defmethod print-method MockMailer
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.mail/-create-mailer :mock
  [m]
  (map->MockMailer m))
