(ns hara.platform.bank.mock
  (:require [hara.platform.bank.mock.instance :as instance]
            [hara.core.base.util :as util]
            [hara.protocol.bank :as protocol.bank]))

(defn op-mock
  "peforms a general operation on the mock
 
   (op-mock (create-mock {:name \"UNION INC.\"})
            instance/add-account
            {:id \"1\" :balances {\"USD\" 100}})
   => (contains {:op :add-account, :id \"1\", :name \"\", :balances {\"USD\" 100}})"
  {:added "1.0"}
  ([{:keys [runtime] :as bank} f {:keys [id] :as op}]
   (let [return (volatile! nil)]
     (swap! runtime
            (fn [instance]
              (let [{:keys [event new]} (f instance op)]
                (vreset! return event)
                new)))
     @return)))

(defn get-account-mock
  "returns account info
 
   (-> (create-mock {:accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (get-account-mock \"1\"))
   => {:id \"1\", :name \"\", :balances {\"USD\" 100}}"
  {:added "1.0"}
  ([{:keys [runtime] :as bank} id]
   (instance/get-account @runtime id)))

(defn list-accounts-mock
  "lists all accounts in the bank
 
   (-> (create-mock {:accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (list-accounts-mock))
   => [\"1\"]"
  {:added "1.0"}
  ([{:keys [runtime] :as bank}]
   (keys (:accounts @runtime))))

(defn add-account-mock
  "adds an account to the bank
 
   (-> (create-mock {})
       (add-account-mock {:id \"1\" :balances {\"USD\" 100}}))
   => (contains {:op :add-account, :id \"1\", :name \"\", :balances {\"USD\" 100}})"
  {:added "1.0"}
  ([bank {:keys [id] :as op}]
   (op-mock bank instance/add-account op)))

(defn remove-account-mock
  "deletes an account from the bank
 
   (-> (create-mock {:accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (remove-account-mock {:id \"1\"}))
   => (contains {:op :remove-account, :id \"1\", :name \"\", :balances {\"USD\" 100}})"
  {:added "1.0"}
  ([bank {:keys [id] :as op}]
   (op-mock bank instance/remove-account op)))

(defn deposit-mock
  "deposits amount into an account
 
   (-> (create-mock {:accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (deposit-mock {:id \"1\" :currency \"USD\" :amount 200}))
   
   => (contains {:op :adjust-account, :id \"1\", :currency \"USD\" :amount 200, :old 100, :new 300})"
  {:added "1.0"}
  ([bank {:keys [id] :as op}]
   (op-mock bank instance/deposit op)))

(defn withdraw-mock
  "withdraws amount from an account
 
   (-> (create-mock {:accounts [{:id \"1\" :balances {\"USD\" 1000}}]})
       (withdraw-mock {:id \"1\" :currency \"USD\" :amount 200}))
   => (contains {:op :adjust-account, :id \"1\", :currency \"USD\" :amount -200, :old 1000, :new 800})"
  {:added "1.0"}
  ([bank {:keys [id] :as op}]
   (op-mock bank instance/withdraw op)))

(defrecord MockBank [name runtime]
  Object
  (toString [_] (str {:name   name
                      :accounts @runtime}))

  protocol.bank/IBank
  (-add-account     [bank data] (add-account-mock bank data))
  (-remove-account  [bank data] (remove-account-mock bank data))
  (-get-account     [bank data] (get-account-mock bank data))
  (-list-accounts   [bank _] (list-accounts-mock bank))
  (-deposit  [bank data] (deposit-mock bank data))
  (-withdraw [bank data] (withdraw-mock bank data)))

(defmethod print-method MockBank
  [v ^java.io.Writer w]
  (.write w (str "#bank.mock" v)))

(defn create-mock
  "creates a bank
 
   (create-mock {:name \"UNION INC.\"})
   ;; #bank{:name \"UNION INC.\", :accounts []}
   => hara.platform.bank.mock.MockBank"
  {:added "1.0"}
  ([{:keys [name accounts]}]
   (MockBank. name
              (atom (instance/create-instance {:accounts accounts})))))

(defmethod protocol.bank/-create :mock
  [m]
  (create-mock m))
