(ns hara.platform.bank.mock.instance
  (:require [hara.core.base.util :as util]))

(defrecord MockBankInstance [accounts log]
  Object
  (toString [_]
    (->> accounts
         (map (fn [[_ {:keys [id balances]}]] [id balances]))
         (sort)
         (vec)
         (str))))

(defmethod print-method MockBankInstance
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn get-account
  "gets account within the instance
   
   (-> (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}]})
       (get-account \"1\"))
   => {:id \"1\", :name \"\", :balances {\"USD\" 10}}"
  {:added "1.0"}
  [instance id]
  (or (get (:accounts instance) id)
      (throw (ex-info "Cannot find account" {:id id}))))

(defn add-account
  "adds an account to the instance
 
   (-> (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}]})
       (add-account {:id \"A\" :name \"test@test.com\" :balances {\"USD\" 10} :tx \"TX\" :time 0}))
   => (contains {:event {:op :create-account, :tx \"TX\", :time 0, :id \"A\", :name \"test@test.com\", :balances {\"USD\" 10}}})"
  {:added "1.0"}
  [instance {:keys [id name balances tx time] :as account
                             :or {id   (util/sid)
                                  name ""
                                  balances 0
                                  tx   (util/sid)
                                  time (System/nanoTime)}}]
  (let [account {:id id :name name :balances balances}
        event (merge {:op :add-account :tx tx :time time} account)
        new-instance (-> instance
                         (update :log conj event)
                         (update :accounts assoc id account))]
    {:event event
     :old instance
     :new new-instance}))

(defn remove-account
  "deletes an account from the instance
 
   (-> (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}]})
       (remove-account {:id \"1\"}))
   => (contains-in {:event {:op :delete-account,
                            :tx string?
                            :time number?,
                            :id \"1\",
                            :name \"\",
                            :balances {\"USD\" 10}}})"
  {:added "1.0"}
  [instance {:keys [id tx time]
             :or {tx   (util/sid)
                  time (System/nanoTime)}}]
  (let [account (get-account instance id)
        event (merge {:op :remove-account :tx tx :time time} account)
        new-instance (-> instance
                         (update :log conj event)
                         (update :accounts dissoc id))]
    {:event event
     :old instance
     :new new-instance}))

(defn deposit
  "deposits amount into the instance
 
   (-> (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}]})
       (deposit {:id \"1\" :currency \"USD\" :amount 100})
       :new
       (get-account \"1\"))
   => {:id \"1\", :name \"\", :balances {\"USD\" 110}}"
  {:added "1.0"}
  [instance {:keys [id currency amount tx time] :as data
             :or {tx   (util/sid)
                  time (System/nanoTime)}}]
  (let [{:keys [balances] :as account} (get-account instance id)
        balance (or (get balances currency) 0)
        event   {:op :adjust-account :id id :tx tx :currency currency :amount amount
                 :time time :old balance :new (+ balance amount)}
        extra   (dissoc data :id :currency :amount :tx :time)
        new-instance (-> instance
                         (update-in [:accounts id :balances currency] (fnil #(+ % amount) 0))
                         (update :log conj (assoc event :extra extra)))]
    {:event event
     :old instance
     :new new-instance}))

(defn withdraw
  "withdraws amount from the instance
 
   (-> (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}]})
       (withdraw {:id \"1\" :currency \"USD\" :amount 100})
       :new
       (get-account \"1\"))
   {:id \"1\", :name \"\", :balance -90}"
  {:added "1.0"}
  [instance {:keys [id currency amount tx time]
             :or {tx   (util/sid)
                  time (System/nanoTime)}
             :as data}]
  (let [{:keys [balances] :as account} (get-account instance id)
        balance (or (get balances currency) 0)
        event   {:op :adjust-account :id id :tx tx :currency currency :amount (- amount)
                 :time time :old balance :new (- balance amount)}
        extra   (dissoc data :id :currency :amount :tx :time)
        new-instance (-> instance
                         (update-in [:accounts id :balances currency] (fnil #(- % amount) 0))
                         (update :log conj (assoc event :extra extra)))]
    {:event event
     :old instance
     :new new-instance}))

(defn create-instance
  "creates a fund instance
 
   (create-instance {:accounts [{:id \"1\" :balances {\"USD\" 10}}
                                {:id \"2\" :balances {\"USD\" 20}}
                                {:id \"3\" :balances {\"USD\" 30}}] :time 0})
   ;; #fund.instance [[\"1\" 10] [\"2\" 20] [\"3\" 30]]
   => hara.platform.bank.mock.instance.MockBankInstance"
  {:added "1.0"}
  [{:keys [accounts time]
                        :or {time (System/nanoTime)}}]
  (:new (reduce (fn [{:keys [new]} account]
                  (add-account new (assoc account :time time)))
                {:new (MockBankInstance. {} [])}
                accounts)))
