(ns hara.platform.cache.atom
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.protocol.component :as protocol.component]
            [hara.print.table :as table]
            [hara.core.base.util :as util]
            [hara.data.base.map :as map]))

(def ^:dynamic *current* nil)

(defn current
  "returns the current time that can be specified for testing
 
   (current)
   ;;=> 1500088974499
 
   (binding [*current* 1000]
     (current))
   => 1000"
  {:added "3.0"}
  ([] (current *current*))
  ([curr]
   (or curr (System/currentTimeMillis))))

(defn set-atom
  "sets the value of an atom"
  {:added "3.0"}
  ([atom key value]
   (doto atom
     (swap! assoc key {:value value})))
  ([atom key value expiry]
   (doto atom
     (swap! assoc key {:value value
                       :expiration (+ (current)
                                      (* expiry 1000))}))))

(defn get-atom
  "sets the value of an atom"
  {:added "3.0"}
  [atom key]
  (let [{:keys [value expiration]} (get @atom key)]
    (if (or (nil? expiration)
            (< (current) expiration))
      value)))

(defn has?-atom
  "contains a key"
  {:added "3.0"}
  [atom key]
  (boolean (get-atom atom key)))

(defn all-atom
  "returns all valid entries in the atom"
  {:added "3.0"}
  [atom]
  (let [current (current)]
    (reduce-kv (fn [m k {:keys [value expiration]}]
                 (if (or (nil? expiration)
                         (< current expiration))
                   (assoc m k value)
                   m))
               {}
               @atom)))

(defn keys-atom
  "returns all valid entries in the atom"
  {:added "3.0"}
  [atom]
  (keys (all-atom atom)))

(defn keys-pattern-atom
  "returns keys that follow a pattern"
  {:added "3.0"}
  [atom ^String pattern]
  (let [pattern (-> pattern
                    (.replaceAll "\\*" ".*")
                    (.replaceAll "\\?" "."))
        regex (re-pattern pattern)]
    (filter (fn [k]
              (re-matches regex (util/keystring k)))
            (keys (all-atom atom)))))

(defn count-atom
  "returns all valid entries in the atom"
  {:added "3.0"}
  [atom]
  (count (keys-atom atom)))

(defn delete-atom
  "returns all valid entries in the atom"
  {:added "3.0"}
  [atom keys]
  (doto atom (swap! #(apply dissoc % keys))))

(defn batch-atom
  "creates a batched operation of inserts and deletes"
  {:added "3.0"}
  [atom add-values add-expiry remove-vec]
  (let [current (current)
        add-map (reduce-kv (fn [out k v]
                             (if-let [exp (get add-expiry k)]
                               (assoc out k {:value v :expiration (+ current
                                                                     (* exp 1000))})
                               (assoc out k {:value v})))
                           {}
                           add-values)]
    (doto atom
      (swap! #(apply dissoc
                     (merge % add-map)
                     remove-vec)))))

(defn clear-atom
  "resets the atom to an empty state"
  {:added "3.0"}
  [atom]
  (doto atom (swap! empty)))

(defn touch-atom
  "extend expiration time if avaliable"
  {:added "3.0"}
  [atom key expiry]
  (doto atom
    (swap! (fn [m] (if-let [{:keys [expiration]} (get m key)]
                     (if (and expiration
                              (< (current) expiration))
                       (assoc-in m [key :expiration]
                                 (+ (current) (* expiry 1000)))
                       m))))))

(defn expired?-atom
  "checks if key is expired"
  {:added "3.0"}
  [atom key]
  (if-let [{:keys [expiration]} (get @atom key)]
    (if (nil? expiration)
      false
      (< expiration (current)))))

(defn expiry-atom
  "checks if key is expired"
  {:added "3.0"}
  [atom key]
  (if-let [{:keys [expiration]} (get @atom key)]
    (cond (nil? expiration)
          :never

          (< expiration (current))
          :expired

          :else
          (quot (- expiration (current)) 1000))))

(defn mget-atom
  "return multiple values"
  {:added "3.0"}
  [atom keys]
  (mapv (partial get-atom atom) keys))

(defn mset-atom
  "sets multiple values"
  {:added "3.0"}
  [atom args]
  (let [count (count args)
        entries-fn (fn [entries] (map/map-vals (fn [v] {:value v}) entries))]
    (cond (zero? count) atom
          (= 1 count) (doto atom
                        (swap! merge (entries-fn (first args))))
          :else (doto atom
                  (swap! merge (entries-fn (into {} (map vec (partition 2 args)))))))))

(defn bulk-atom
  "groups multiple operations"
  {:added "3.0"}
  [atom thunk]
  (thunk))

(defn transact-atom
  "transacts multiple operations (single thread workiing)"
  {:added "3.0"}
  [atom thunk]
  (thunk))

(defn calc-index
  "allows reverse indices"
  {:added "3.0"}
  [coll i]
  (let [len   (count coll)]
    (if (neg? i) (+ len i) i)))

(defn lindex-atom
  "retrieves list value at key"
  {:added "3.0"}
  [atom key index]
  (let [arr (get-atom atom key)]
    (get arr index)))

(defn llen-atom
  "returns length of list at key"
  {:added "3.0"}
  [atom key]
  (count (get-atom atom key)))

(defn lpop-atom
  "removes head of list"
  {:added "3.0"}
  [atom key]
  (let [arr (get-atom atom key)]
    (set-atom atom key (subvec arr 1))))

(defn lpush-atom
  "adds to head of list in order"
  {:added "3.0"}
  [atom key elements]
  (let [arr (or (get-atom atom key) [])]
    (set-atom atom key (vec (concat (reverse elements) arr)))))

(defn lrange-atom
  "returns items in a given range"
  {:added "3.0"}
  [atom key start stop]
  (let [arr   (get-atom atom key)
        start (calc-index arr start)
        stop (calc-index arr stop)]
    (->> (drop start arr)
         (take (- (inc stop) start)))))

(defn lrem-atom
  "removes entries equal to the value"
  {:added "3.0"}
  [atom key count value]
  (let [arr   (get-atom atom key)
        lrem  (fn [count arr]
                (loop [acc 0
                       output [] 
                       [x & more :as arr] arr]
                  (cond (empty? arr) output
                        (= count acc) (vec (concat output arr))
                        (= value x) (recur (inc acc) output more)
                        :else (recur acc (conj output x) more))))
        arr   (cond (zero? count)
                    (vec (remove #(= value %) arr))
                    
                    (pos? count)
                    (lrem count arr)

                    (neg? count)
                    (vec (reverse (lrem (- count) (reverse arr)))))]
    (set-atom atom key arr)))

(defn lset-atom
  "sets entry at index"
  {:added "3.0"}
  [atom key index element]
  (let [arr (get-atom atom key)
        index (calc-index arr index)]
    (set-atom atom key (assoc arr index element))))

(defn ltrim-atom
  "keeps the items in range"
  {:added "3.0"}
  [atom key start stop]
  (let [arr   (get-atom atom key)
        start (calc-index arr start)
        stop (calc-index arr stop)]
    (set-atom atom key (subvec arr start (inc stop)))))

(defn rpop-atom
  "removes from tail of list"
  {:added "3.0"}
  [atom key]
  (let [arr (get-atom atom key)]
    (set-atom atom key (pop arr))))

(defn rpush-atom
  "adds to tail of list in order"
  {:added "3.0"}
  [atom key elements]
  (let [arr (or (get-atom atom key) [])]
    (set-atom atom key (apply conj arr elements))))

(defn hdel-atom
  "removes a field"
  {:added "3.0"}
  [atom key fields]
  (let [m (or (get-atom atom key) {})]
    (set-atom atom key (apply dissoc m fields))))

(defn hexists-atom
  "checks if field exists"
  {:added "3.0"}
  [atom key field]
  (contains? (get-atom atom key) field))

(defn hget-atom
  "returns the value of field"
  {:added "3.0"}
  [atom key field]
  (get (get-atom atom key) field))

(defn hmget-atom
  "returns values of multiple fields"
  {:added "3.0"}
  [atom key fields]
  (map (get-atom atom key) fields))

(defn hgetall-atom
  "return values of all fields"
  {:added "3.0"}
  [atom key]
  (mapcat identity (get-atom atom key)))

(defn hkeys-atom
  "return all keys"
  {:added "3.0"}
  [atom key]
  (keys (get-atom atom key)))

(defn hlen-atom
  "returns number of items"
  {:added "3.0"}
  [atom key]
  (count (get-atom atom key)))

(defn hset-atom
  "sets a field"
  {:added "3.0"}
  ([atom key input]
   (let [m (or (get-atom atom key) {})]
     (cond (= (count input) 1)
           (set-atom atom key (merge m (first input)))

           :else
           (set-atom atom key (apply assoc m input))))))

(defn hvals-atom
  "return field values"
  {:added "3.0"}
  [atom key]
  (vals (get-atom atom key)))

(defn zadd-atom
  "adds items with scores"
  {:added "3.0"}
  ([atom key input]
   (let [m (or (get-atom atom key) {})]
     (cond (= (count input) 1)
           (set-atom atom key (merge m (map/map-entries (fn [[k v]] [v k])  (first input))))
           
           :else
           (set-atom atom key (apply assoc m (reverse input)))))))

(defn zcard-atom
  "number of entries in set"
  {:added "3.0"}
  ([atom key]
   (count (get-atom atom key))))

(defn zcount-atom
  "counts number of items within a score range"
  {:added "3.0"}
  [atom key min max]
  (let [m (or (get-atom atom key) {})]
    (count (filter (fn [v] (<= min v max)) (vals m)))))

(defn zpopmin-atom
  "kicks out the smallest score"
  {:added "3.0"}
  [atom key count]
  (let [m (or (get-atom atom key) {})
        results (take count (sort-by val m))]
    (set-atom atom key (apply dissoc m (map first results)))
    (flatten results)))

(defn zpopmax-atom
  "kicks out the largest score"
  {:added "3.0"}
  [atom key count]
  (let [m (or (get-atom atom key) {})
        results (take count (reverse (sort-by val m)))]
    (set-atom atom key (apply dissoc m (map first results)))
    (flatten results)))

(defn zrange-atom
  "returns range by order put in"
  {:added "3.0"}
  [atom key start stop]
  (let [m (or (get-atom atom key) {})
        start (calc-index m start)
        stop  (calc-index m stop)]
    (->> (sort-by val m)
         (drop start)
         (take (inc (- stop start)))
         (map first))))

(defn zrangebyscore-atom
  "finds all items by score"
  {:added "3.0"}
  [atom key min max]
  (let [m (or (get-atom atom key) {})]
    (->> (sort-by val m)
         (filter (fn [[k v]]
                   (<= min v max)))
         (map first))))

(defn zrank-atom
  "gets the ranking of an item"
  {:added "3.0"}
  [atom key member]
  (let [m (or (get-atom atom key) {})
        scores (sort (set (vals m)))
        val (get m member)]
    (.indexOf ^java.util.List scores val)))

(defn zrem-atom
  "remove items"
  {:added "3.0"}
  [atom key members]
  (let [m (or (get-atom atom key) {})]
    (set-atom atom key (apply dissoc m members))))

(defn zremrangebyscore-atom
  "removes range by score"
  {:added "3.0"}
  [atom key min max]
  (let [m (or (get-atom atom key) {})
        ids (->> (filter (fn [[k v]] (<= min v max)) m)
                 (map first))]
    (set-atom atom key (apply dissoc m ids))))

(defn zrevrange-atom
  "return reverse range"
  {:added "3.0"}
  [atom key start stop]
  (let [m (or (get-atom atom key) {})
        start (calc-index m start)
        stop (calc-index m stop)]
    (->> (reverse (sort-by val m))
         (drop start)
         (take (inc (- stop start)))
         (map first))))

(defn zrevrank-atom
  "return reverse rank"
  {:added "3.0"}
  [atom key member]
  (let [m (or (get-atom atom key) {})
        scores (reverse (sort (set (vals m))))
        val (get m member)]
    (.indexOf ^java.util.List scores val)))

(defn zscore-atom
  "returns item score"
  {:added "3.0"}
  [atom key member]
  (let [m (or (get-atom atom key) {})]
    (get m member)))

(extend-protocol protocol.cache/ICache

  clojure.lang.Atom
  (-set
    ([atom key value] (set-atom atom key value))
    ([atom key value expiry] (set-atom atom key value expiry)))

  (-get      [atom key] (get-atom atom key))
  (-has?     [atom key] (has?-atom atom key))
  (-count    [atom] (count-atom atom))
  (-batch    [atom add-values add-expiry remove-vec] (batch-atom atom add-values add-expiry remove-vec))
  (-delete   [atom keys] (delete-atom atom keys))
  (-clear    [atom] (clear-atom atom))
  (-all      [atom] (all-atom atom))
  (-keys
    ([atom] (keys-atom atom))
    ([atom pattern] (keys-pattern-atom atom pattern)))
  (-touch    [atom key expiry] (touch-atom atom key expiry))
  (-expired? [atom key] (expired?-atom atom key))
  (-expiry   [atom key] (expiry-atom atom key)))

(extend-protocol protocol.cache/ICacheExtend
  clojure.lang.Atom
  (-mget     [atom keys] (mget-atom atom keys))
  (-mset     [atom args] (mset-atom atom args))
  (-bulk     [atom thunk] (bulk-atom atom thunk))
  (-transact [atom thunk] (transact-atom atom thunk)))

(extend-protocol protocol.cache/ICacheList
  clojure.lang.Atom
  (-lindex   [atom key index] (lindex-atom atom key index))
  (-llen     [atom key] (llen-atom atom key))
  (-lpop     [atom key] (lpop-atom atom key))
  (-lpush    [atom key elements] (lpush-atom atom key elements))
  (-lrange   [atom key start stop] (lrange-atom atom key start stop))
  (-lrem     [atom key count value] (lrem-atom atom key count value))
  (-lset     [atom key index element] (lset-atom atom key index element))
  (-ltrim    [atom key start stop] (ltrim-atom atom key start stop))
  (-rpop     [atom key] (rpop-atom atom key))
  (-rpush    [atom key elements] (rpush-atom atom key elements)))

(extend-protocol protocol.cache/ICacheHash
  clojure.lang.Atom
  (-hdel     [atom key fields] (hdel-atom atom key fields))
  (-hexists  [atom key field] (hexists-atom atom key field))
  (-hget     [atom key field] (hget-atom atom key field))
  (-hmget    [atom key fields] (hmget-atom atom key fields))
  (-hgetall  [atom key] (hgetall-atom atom key))
  (-hkeys    [atom key] (hkeys-atom atom key))
  (-hlen     [atom key] (hlen-atom atom key))
  (-hset     [atom key args] (hset-atom atom key args))
  (-hvals    [atom key] (hvals-atom atom key)))

(extend-protocol protocol.cache/ICacheSortedSet
  clojure.lang.Atom
  (-zadd     [atom key args] (zadd-atom atom key args))
  (-zcard    [atom key] (zcard-atom atom key))
  (-zcount   [atom key min max] (zcount-atom atom key min max))
  (-zpopmin  [atom key count] (zpopmin-atom atom key count))
  (-zpopmax  [atom key count] (zpopmax-atom atom key count))
  (-zrange   [atom key start stop] (zrange-atom atom key start stop))
  (-zrangebyscore [atom key min max] (zremrangebyscore-atom atom key min max)) 
  (-zrank    [atom key member] (zrank-atom atom key member))
  (-zrem     [atom key members] (zrem-atom atom key members))
  (-zremrangebyscore [atom key min max] (zremrangebyscore-atom atom key min max)) 
  (-zrevrange [atom key start stop] (zrevrange-atom atom key start stop))
  (-zrevrank  [atom key member] (zrevrank-atom  atom key member))
  (-zscore    [atom key member] (zscore-atom  atom key member)))
