(ns hara.platform.cache.log
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.protocol.log :as protocol.log]
            [hara.protocol.component :as protocol.component]
            [hara.core.component :as component]
            [hara.log.core :as core]
            [hara.io.concurrent :as cc]))

(defn logger-process-cache
  "process the cache logger"
  {:added "3.0"}
  ([{:keys [cache publish] :as logger} items]
   (protocol.cache/-rpush cache publish (mapv prn-str items))))

(defrecord CacheLogger [instance]
  
  Object
  (toString [logger] (str "#log.cache" (core/logger-info-core logger)))

  protocol.log/ILogger
  (-logger-write [logger entry] (core/logger-write-core logger entry))
  
  protocol.log/ILoggerProcess
  (-logger-process [logger items] (logger-process-cache logger items))

  protocol.component/IComponent
  (-start [{:keys [connect] :as logger}]
    (let [cache (if connect
                  (-> (protocol.cache/-create connect)
                      (component/start)))]
      (cond-> logger
        cache (assoc :cache cache))))

  (-stop [{:keys [connect cache] :as logger}]
    (let [_  (if connect
               (component/stop cache))]
      (cond-> logger
        connect (dissoc :cache)))))

(defmethod print-method CacheLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :cache
  [m]
  (let [queue (cc/queue)]
    (-> (merge {:interval 200
                :max-batch 1000
                :queue queue
                :executor (cc/single-executor 1)}
               m)
        (core/logger-init-core)   
        (map->CacheLogger))))

(defn cache-logger
  "creates a cache logger
 
   (cache-logger {:interval 500 :max-batch 10000})"
  {:added "3.0"}
  ([] (cache-logger nil))
  ([m]
   (-> (protocol.log/-create (assoc m :type :cache))
       (component/start))))

