(ns hara.platform.eft.mock
  (:require [hara.platform.bank :as bank]
            [hara.protocol.eft :as protocol.eft]))

(def ^:dynamic *bank*
  (bank/bank {:type :mock
              :name "TEST"
              :accounts [{:id "test@test.com" :balances {}}
                         {:id "account0@test.com" :balances {}}
                         {:id "account1@test.com" :balances {}}
                         {:id "account2@test.com" :balances {}}
                         {:id "account3@test.com" :balances {}}]}))

(defn deposit-mock
  ([{:keys [bank] :as fund} {:keys [id currency amount] :as data}]
   (try
     (let [{:keys [tx time]} (bank/deposit bank data)]
       {:status :success
        :data {:type :fund.deposit :receipt tx :time time :currency currency :amount amount :id id}})
     (catch clojure.lang.ExceptionInfo ex
       {:status :error
        :tag :error/transfer
        :message (.getMessage ex)
        :data (ex-data ex)}))))

(defn withdraw-mock
  ([{:keys [bank] :as fund} {:keys [id currency amount] :as data}]
   (try
     (let [{:keys [tx time]} (bank/withdraw bank data)]
       {:status :success
        :data {:type :fund.withdraw :receipt tx :time time :currency currency :amount amount :id id}})
     (catch clojure.lang.ExceptionInfo ex
       {:status :error
        :tag :error/transfer
        :message (.getMessage ex)
        :data (ex-data ex)}))))

(defn submit-mock
  "submits a transaction to eft
 
   (binding [mock/*bank* (bank/bank {:type :mock :accounts [{:id \"A\" :balances {\"USD\" 1000}}]})]
     (-> (eft/create {:type :mock})
         (submit-mock {:type :deposit :id \"A\" :currency \"USD\" :amount 500})))
   => (contains-in {:status :success,
                    :data {:type :fund.deposit,
                           :receipt string?
                           :time number?
                           :amount 500}})"
  {:added "3.0"}
  ([{:keys [bank] :as fund} {:keys [type amount] :as data}]
   (case type
     :deposit  (deposit-mock fund data)
     :withdraw (withdraw-mock fund data))))

(defn status-mock
  "gets the status of a given transaction
 
   (binding [mock/*bank* (bank/bank {:type :mock :accounts [{:id \"A\" :balances {\"USD\" 1000}}]})]
     (-> (eft/create {:type :mock})
        (status-mock \"NOT-FOUND\"))
     => :not-found)"
  {:added "3.0"}
  ([{:keys [bank] :as fund} receipt]
   (let [log (:log @(:runtime *bank*))]
     (if (contains? (set (map :tx log)) receipt)
       :processed
       :not-found))))

(defn detail-mock
  "looks into the actual transaction
   
   (binding [mock/*bank* (bank/bank {:type :mock :accounts [{:id \"A\" :balances {\"USD\" 1000}}]})]
     (-> (eft/create {:type :mock})
         (detail-mock (-> @(:runtime mock/*bank*) :log first :tx))))
   => (contains {:op :add-account, :id \"A\", :name \"\", :balances {\"USD\" 1000}})"
  {:added "3.0"}
  ([{:keys [bank] :as fund} receipt]
   (let [log (:log @(:runtime *bank*))]
     (first (filter #(-> % :tx (= receipt)) log)))))

(defrecord MockFund [bank]
  Object
  (toString [m] (str (into {} m)))
  
  protocol.eft/IFund
  (-submit       [fund data] (submit-mock fund data))
  (-status       [fund receipt] (status-mock fund receipt))
  (-detail       [fund receipt] (detail-mock fund receipt)))

(defmethod print-method MockFund
  [v ^java.io.Writer w]
  (.write w (str "#fund.mock" v)))

(defmethod protocol.eft/-create :mock
  [{:keys [bank] :as m}]
  (cond-> (map->MockFund m)
    (nil? bank) (assoc :bank *bank*)))

(comment

  
  
  (bank/deposit *bank* {:id "test@test.com"
                        :currency "USD"
                        :amount 1000})
  (:log @(:runtime *bank*))
  
  
  
  
  )
