(ns hara.platform.exchange.consumer
  (:require [hara.protocol.component :as protocol.component]
            [hara.protocol.exchange :as protocol.exchange]
            [hara.core.component :as component]
            [hara.io.concurrent :as cc]
            [hara.log :as log]))

(defn wrap-logger [handler logger]
  (fn [body headers opts]
    (if logger
      (log/with-logger logger
        (handler body headers opts))
      (handler body headers opts))))

(defn wrap-pool [handler pool]
  (cond pool
        (fn [body headers opts]
          (cc/submit (:executor pool)
                     (fn []
                       (try
                         (handler body headers opts)
                         (catch Throwable t
                           (log/error t))))))
        
        :else
        handler))

(defn create-pool [{:keys [size max keep-alive]
                    :or {size 10 max 20 keep-alive 1000}}]
  {:executor (cc/pool-executor size max keep-alive)})

(defrecord Consumer [exchange logger]
  protocol.component/IComponent
  (-start [{:keys [key name pool queue construct instance] :as service}]
    (log/status logger (str (or name "Consumer") " started")
                #:consumer{:key key :queue queue})
    (let [pool (if pool (create-pool pool))]
      (reset! instance pool)
      (protocol.exchange/-add-consumer exchange queue
                                       {:id key
                                        :function (-> (construct service)
                                                      (wrap-logger logger)
                                                      (wrap-pool pool))})
      service))
  (-stop  [{:keys [key name queue instance] :as service}]
    (reset! instance nil)
    (log/status logger (str (or name "Consumer") " stopped")
                #:consumer{:key key :queue queue})
    (protocol.exchange/-delete-consumer exchange queue key)
    service))

(defn create
  "creates an uninitialised history service
 
   (create {:store    (table/datastore   {:type :mock})
            :exchange (exchange/exchange {:type :mock})})"
  {:added "0.1"}
  [m]
  (map->Consumer (assoc m :instance (atom nil))))

(defn consumer
  "creates an inititialised history service
   
   (def -service-
     (history-service
      {:store     (table/datastore   {:type :mock :schema schema/*schema*})
       :exchange  (exchange/exchange {:type :mock :routing -routing-})}))"
  {:added "0.1"}
  [m]
  (-> (create m)
      (component/start)))
