(ns hara.platform.keystore
  (:require [hara.protocol.keystore :as protocol.keystore]
            [hara.platform.keystore.base]
            [hara.core.component :as component]))

(defn put-in
  "adds data to the database
 
   (-> (keystore)
       (put-in [:b] {:d 2})
       (put-in [:b] {:c 2})
       (peek-in))
   => {:b {:d 2, :c 2}}"
  {:added "3.0"}
  ([db v] (put-in db [] v))
  ([db path v]
   (protocol.keystore/-put-in db path v)))

(defn peek-in
  "accesses all or part of the database
 
   (-> (keystore)
       (put-in {:a 1, :b {:c 2}})
       (peek-in [:b]))
   => {:c 2}"
  {:added "3.0"}
  ([db] (peek-in db []))
  ([db path]
   (protocol.keystore/-peek-in db path)))

(defn keys-in
  "returns all keys in the particular level
 
   (-> (keystore)
       (put-in {:a 1, :b {:c 2}})
       (keys-in)
       (sort))
   => [:a :b]"
  {:added "3.0"}
  ([db] (keys-in db []))
  ([db path]
   (protocol.keystore/-keys-in db path)))

(defn drop-in
  "removes keys in the database
   
   (-> (keystore)
       (put-in {:a 1, :b {:c 2}})
       (drop-in [:a])
       (peek-in))
   => {:b {:c 2}}"
  {:added "3.0"}
  ([db] (drop-in db []))
  ([db path]
   (protocol.keystore/-drop-in db path)))

(defn set-in
  "adds data to the database. overwrites
 
   (-> (keystore)
       (set-in [:b] {:d 2})
       (set-in [:b] {:c 2})
       (peek-in))
   => {:b {:c 2}}"
  {:added "3.0"}
  ([db v] (set-in db [] v))
  ([db path v]
   (protocol.keystore/-set-in db path v)))

(defn select-in
  "selects the key values corresponding to the query
   (-> (keystore)
       (set-in [:data] {:a 1
                        :b 2
                        :c 3
                        :d 4})
       (select-in [:data] odd?))
   => [[:a 1] [:c 3]]"
  {:added "3.0"}
  ([db v] (select-in db [] v))
  ([db path v]
   (protocol.keystore/-select-in db path v)))

(defn batch-in
  "batch add and removal of data
 
   (-> (keystore)
       (put-in {:data {:a 1
                       :b 2}
                :input {:c 3
                        :d 4}})
       (batch-in {:data {:x 1}
                  :input {:y 2}}
                 [[:data :a] [:input :d]])
      (peek-in))
   => {:data {:b 2, :x 1}
       :input {:c 3, :y 2}}"
  {:added "3.0"}
  ([db add-map]
   (batch-in db [] add-map []))
  ([db add-map remove-vec]
   (batch-in db [] add-map remove-vec))
  ([db path add-map remove-vec]
   (protocol.keystore/-batch-in db path add-map remove-vec)))

(defn create
  "creates a keystore that is component compatible
 
   (keystore/create {:type :mock
                     :file {:path \"dev/scratch/test.edn\"
                            :reset true}})
   ;;=> #keystore.mock<uninitiased>
   "
  {:added "3.0"}
  [m]
  (protocol.keystore/-create m))

(defn keystore
  "creates an active keystore
 
   (keystore {:type :mock
              :initial {:hello :world}
              :file {:path \"dev/scratch/test.edn\"
                     :reset true
                     :cleanup true}})
   ;;=> #keystore.mock{:hello :world}
   "
  {:added "3.0"}
  ([]
   (keystore {:type :mock}))
  ([m]
   (-> (create m)
       (component/start))))
