(ns hara.log.form
  (:require [hara.data.base.map :as map]
            [hara.core.base.util :as util]
            [hara.string :as string]
            [clojure.main :as main]
            [hara.log.common :as common]
            [hara.log.core :as core]))

(defn log-meta
  "gets the metadata information on macroexpand"
  {:added "3.0"}
  ([form]
   (let [{:keys [line column]} (meta form)
         ns    (str (.getName *ns*))]
     {:log/namespace ns
      :log/line      line
      :log/column    column})))

(defn log-function-name
  "gets the function name
 
   (log-function-name \"ns$function_name\")
   => \"function-name\""
  {:added "3.0"}
  [name]
  (-> (string/split name #"\$")
      (second)
      (main/demunge)))

(defn log-runtime-raw
  "returns the runtime information of the log entry
 
   (set (keys (log-runtime-raw nil \"hara.log.form-test\")))
   => #{:log/function :log/method :log/filename}"
  {:added "3.0"}
  [id ns]
  (let [e        (RuntimeException. "")
        frames   (.getStackTrace e)]
    (comment {:log/return (.printStackTrace e)})
    (->> frames
         (keep (fn [^StackTraceElement frame]
                 (let [name (.getClassName frame)]
                   (if (and (not (.startsWith name "hara.log.form$log_runtime"))
                            (.startsWith name (munge ns))
                            (.contains name "$"))
                     {:log/function (log-function-name name)
                      :log/method   (.getMethodName frame)
                      :log/filename (.getFileName frame)}))))
         first)))

(def log-runtime (memoize log-runtime-raw))

(defn log-check
  "checks to see if form should be ignored
 
   (log-check :debug)
   => true"
  {:added "3.0"}
  [level]
  (if (or (not common/*static*)
          (>= (get common/+levels+ level)
              (get common/+levels+ common/*level*)))
    true false))

(defn log-form
  "function for not including log forms on static compilation"
  {:added "3.0"}
  ([form logger level message data]
   (if (log-check level)
     (let [id   (util/sid)
           meta (log-meta form)]
       `(core/log-write ~logger ~level
                        (merge ~meta (log-runtime ~id ~(:log/namespace meta)))
                        ~message ~data)))))

(defmacro log
  "produces a log"
  {:added "3.0"}
  ([data]
   (log-form &form `common/*logger* :info "" data))
  ([message data]
   (log-form &form `common/*logger* :info message data))
  ([level message data]
   (log-form &form `common/*logger* level message data))
  ([logger level message data]
   (log-form &form logger level message data)))

(defn log-macro-form
  "creates a standard form given a type"
  {:added "3.0"}
  [type]
  (list 'defmacro (symbol (name type))
        (list '[data]
              (list 'log-form '&form (list 'quote 'hara.log.common/*logger*) type "" 'data))
        (list '[message data]
              (list 'log-form '&form (list 'quote 'hara.log.common/*logger*) type 'message 'data))
        (list '[logger message data]
              (list 'log-form '&form 'logger type 'message 'data))))

(defmacro log-macros
  "creates forms for `debug`, `status`, `info`, `error`, `warn`, `fatal`"
  {:added "3.0"}
  [types]
  (mapv log-macro-form types))

(log-macros [:note :debug :code :status :info :todo :warn :error :fatal])

(defn spy-form
  "helper function for the `spy` macro"
  {:added "3.0"}
  ([form logger data body]
   (cond (not (log-check :spy))
         body

         :else
         (let [id   (util/sid)
               meta (log-meta form)
               message (str "FORM " (pr-str body))]   
           `(let [out# ~body]
              (core/log-write ~logger :spy (merge ~meta (log-runtime ~id ~(:log/namespace meta)))
                              ~message (merge ~data {:log/return out#}))
              out#)))))

(defmacro spy
  "outputs the return for the number"
  {:added "3.0"}
  ([body]
   (spy-form &form `common/*logger* nil body))
  ([data body]
   (spy-form &form `common/*logger* data body))
  ([logger data body]
   (spy-form &form logger data body)))

(defn trace-form
  "helper function for the `trace` macro"
  {:added "3.0"}
  [form logger data body]
  (cond (not (log-check :trace))
        body

        :else
        (let [id   (util/sid)
              meta (log-meta form)
              message (str "TIME " (pr-str body))]
          `(let [t#  (System/currentTimeMillis)
                 t0# (System/nanoTime)]
             (try
               (let [out# ~body]
                 (core/log-write ~logger :trace
                                 (merge ~meta (log-runtime ~id ~(:log/namespace meta)))
                                 ~message
                                 (assoc ~data
                                        :log/duration (- (System/nanoTime) t0#)
                                        :log/outcome :ok
                                        :log/return out#)
                                 t#)
                 out#)
               (catch Exception e#
                 (core/log-write ~logger :trace
                                 (merge ~meta (log-runtime ~id ~(:log/namespace meta)))
                                 ~message
                                 (assoc ~data
                                        :log/duration (- (System/nanoTime) t0#)
                                        :log/outcome :error
                                        :log/exception e#)
                                 t#)
                 (throw e#)))))))

(defmacro trace
  "outputs and times the call
 
   (binding [common/*level* :debug
             common/*logger* (core/identity-logger {:type :identity :level :debug})]
     (trace (+ 1 2 3 4)))
   => 10"
  {:added "3.0"}
  ([body]
   (trace-form &form `common/*logger* nil body))
  ([data body]
   (trace-form &form `common/*logger* data body))
  ([logger data body]
   (trace-form &form logger data body)))
