(ns hara.platform.mail.interop.message
  (:require [hara.object :as object]
            [hara.platform.mail.interop.address :as address]
            [hara.platform.mail.interop.body :as body]
            [hara.platform.mail.interop.header :deps true]
            [hara.platform.mail.interop.flag :deps true]
            [hara.platform.mail.interop.recipient :as recipient]
            [hara.platform.mail.interop.wrap :as wrap])
  (:import [javax.mail Message Session]
           [javax.mail.internet MimeMessage MimeMultipart]))

(defn empty-message
  "creates an empty MimeMessage
 
   (object/to-data (empty-message))
   => {:all-headers [],
       :all-header-lines [],
       :size -1,
       :content-type \"text/plain\",
       :flags [],
       :line-count -1}"
  {:added "3.0"}
  ([]
   (empty-message (Session/getInstance (java.util.Properties.))))
  ([^Session session]
   (MimeMessage. session)))

(defn get-recipients
  "gets recipients of a particular field
   
   (-> (doto (empty-message)
         (.addRecipient (recipient/list :to)
                        (InternetAddress. \"z@z.com\")))
       ((get-recipients :to))
       (seq))
   ;;=> (#address \"z@z.com\")
   "
  {:added "3.0"}
  [k]
  (fn [^Message m] (.getRecipients m (recipient/list k))))

(defn set-recipients
  "sets recipients of a particular field
 
   (-> (doto (empty-message)
         ((set-recipients :to)  [\"a@a.com\"])
         ((set-recipients :bcc) [\"b@b.com\"])
         ((set-recipients :cc)  [\"c@c.com\"]))
       (object/to-data)      :all-recipients)
   => [\"a@a.com\" \"c@c.com\" \"b@b.com\"]"
  {:added "3.0"}
  [k]
  (fn [^Message m l]
    (doto m (.addRecipients (recipient/list k)
                            (address/write-addresses l)))))

(defn set-body
  "sets the body of a given message
 
   (-> (set-body (empty-message) \"Hello World\")
       (object/get :body))
   => \"Hello World\""
  {:added "3.0"}
  [^Message m v]
  (cond (string? v)
        (.setText m v)

        (or (instance? MimeMultipart v) (vector? v))
        (.setContent m (body/body v))

        :else
        (throw (Exception. "Entry either text or a multipart entry")))
  m)

(def read-mime-message
  (let [methods (-> (object/read-getters MimeMessage)
                    (dissoc :content :data-handler :input-stream :raw-input-stream :content-stream)
                    (update-in [:from :fn] wrap/wrap-first)
                    (merge {:body {:type Object
                                   :fn (wrap/wrap-suppress (fn [^Message m]
                                                          (.getContent m)))}
                            :to   {:type java.util.List
                                   :fn (get-recipients :to)}
                            :cc   {:type java.util.List
                                   :fn (get-recipients :cc)}
                            :bcc  {:type java.util.List
                                   :fn (get-recipients :bcc)}}))]
    methods))

(def write-mime-message
  (let [{:keys [content text] :as methods} (object/write-all-setters MimeMessage {:select {:from java.lang.String}})]
    (merge (dissoc methods :content :text)
           {:body {:type Object
                   :fn set-body}
            :to  {:type java.util.List
                  :fn (set-recipients :to)}
            :cc  {:type java.util.List
                  :fn (set-recipients :cc)}
            :bcc {:type java.util.List
                  :fn (set-recipients :bcc)}})))

(object/map-like
 MimeMessage
 {:tag "message"
  :read  {:methods read-mime-message}
  :write {:methods write-mime-message
          :empty (fn [] (empty-message))}})

(defn ^MimeMessage message
  "constructs a MimeMessage from a map
 
   (-> (message {:from \"z@z.com\"
                 :cc  [\"a@a.com\"]
                 :bcc [\"b@b.com\" \"c@c.com\"]
                 :reply-to [\"y@y.com\"]
                 :subject \"Hello There\"
                 :body \"This is Cool\"})
       (object/to-data)
       (select-keys [:all-headers :subject :body]))
   => {:all-headers [[\"From\" \"z@z.com\"]
                     [\"Reply-To\" \"y@y.com\"]
                     [\"Cc\" \"a@a.com\"]
                     [\"Bcc\" \"b@b.com, c@c.com\"]
                    [\"Subject\" \"Hello There\"]]
       :subject \"Hello There\"
       :body \"This is Cool\"}
 
   "
  {:added "3.0"}
  [{:keys [from to cc bcc subject content text reply-to] :as m}]
  (object/from-data m MimeMessage))
