(ns hara.platform.server
  (:require [hara.protocol.server :as protocol.server]
            [hara.protocol.component :as protocol.component]
            [hara.core.component :as component]
            [hara.log :as log]))

(defn make-handler
  "makes a handler given function passed in"
  {:added "3.0"}
  [{:keys [routes dev] :as server}]
  (if dev
    (routes server)
    (fn [req]
      ((routes server) req))))

(defn start-server
  "starts the server"
  {:added "3.0"}
  ([{:keys [type name handler host port silent] :as server}]
   (let [handler  (or handler (make-handler server))
         start-fn (protocol.server/-start-fn server)
         instance (log/trace (start-fn handler server))
         _  (reset! (:instance server) instance)]
     (log/status (str (or name "Server")
                    " started: http://" host ":" port)
               #:server{:type type :name name})
     server)))

(defn stop-server
  "stops the server"
  {:added "3.0"}
  [{:keys [type name host port silent] :as server}]
  (let [stop-fn (protocol.server/-stop-fn {:type type})]
    (log/trace (stop-fn @(:instance server)))
    (reset! (:instance server) nil)
    (log/status (str (or name "Server")
                     " stopped: http://" host ":" port)
                #:server{:type type :name name})
    server))

(defrecord Server [type host port]
  Object
  (toString [server]
    (str "#server." (name type) {:port port}))
  
  protocol.component/IComponent
  (-start [server] (start-server server))
  (-stop  [server] (stop-server server))
  (-started? [server] (boolean @(:instance server))))

(defmethod print-method Server
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn create
  "makes a component compatible server"
  {:added "3.0"}
  [m]
  (map->Server (-> (merge {:port 1984 :host "0.0.0.0"} 
                          m)
                   (assoc :instance (atom {})))))

(defn server
  "creates a running server"
  {:added "3.0"}
  [m]
  (-> (create m)
      (component/start)))
