(ns hara.platform.storage.mock
  (:require [hara.protocol.storage :as protocol.storage]
            [hara.protocol.component :as protocol.component])
  (:import (java.io ByteArrayInputStream)))

(defn write-atom
  "writes a key to the atom
 
   (write-atom (atom {}) \"foo\" (.getBytes \"Bar Baz\"))
   => (contains-in {\"foo\" {:data bytes?, :time number?}})"
  {:added "0.1"}
  [atom key bytes]
  (swap! atom assoc key {:data bytes :time (System/currentTimeMillis)}))

(defn read-atom
  "reads value from a key
 
   (-> (component/with [storage (storage/create -conf-)]
         (read-atom (:state storage) \"hello\"))
       slurp)
   => \"Hello World\""
  {:added "0.1"}
  [atom key]
  (if-let [{:keys [data]} (get @atom key)]
    (ByteArrayInputStream. data)))

(defn list-atom
  "lists all keys in the storage
 
   (component/with [storage (storage/create -conf-)]
     (list-atom (:state storage)))
   => [\"hello\" \"sample\"]"
  {:added "0.1"}
  [atom]
  (keys @atom))

(defn info-atom
  "gets an info for a given key
 
   (component/with [storage (storage/create -conf-)]
     (info-atom (:state storage) \"sample\"))
   => (contains {:time number? :size 11})"
  {:added "0.1"}
  [atom key]
  (if-let [{:keys [time data]} (get @atom key)]
    {:time time :size (alength data)}))

(defn clear-atom
  "clears the storage or a key
 
   (component/with [storage (storage/create -conf-)]
     (clear-atom (:state storage) \"sample\")
     (list-atom (:state storage)))
   => [\"hello\"]
 
   (component/with [storage (storage/create -conf-)]
     (clear-atom (:state storage))
     (list-atom (:state storage)))
   => nil"
  {:added "0.1"}
  ([atom]
   (reset! atom {}))
  ([atom key]
   (swap! atom dissoc key)))

(defrecord MockStorage [state]
  Object
  (toString [storage]
    (str {:items (count (list-atom state))}))
  
  protocol.storage/IStorage
  (-write    [storage key bytes] (write-atom state key bytes) storage)
  (-read     [storage key] (read-atom state key))
  (-list     [storage] (list-atom state))
  (-info     [storage key] (info-atom state key))
  (-clear    [storage] (clear-atom state) storage)
  (-clear    [storage key] (clear-atom state key) storage)

  protocol.component/IComponent
  (-start [{:keys [initial] :as storage}]
    (doseq [[k bytes] initial]
      (write-atom state k bytes))
    storage)
  (-stop [storage] storage))

(defmethod print-method MockStorage
  ([v ^java.io.Writer w]
   (.write w (str #"storage.mock " v))))

(defmethod protocol.storage/-create :mock
  ([m]
   (map->MockStorage (assoc m :state (atom {})))))

