(ns hara.platform.storage.s3
  (:require [hara.protocol.storage :as protocol.storage]
            [hara.protocol.component :as protocol.component]
            [cognitect.aws.client.api :as aws]
            [cognitect.aws.credentials :as credentials]))

(defn write-s3
  "writes to storage given key and bytes
 
   (component/with [storage (storage/create -conf-)]
     (write-s3 storage \"foo\" (.getBytes \"Bar Baz\")))
   => (contains {:ETag string?})"
  {:added "0.1"}
  [{:keys [bucket client]} key bytes]
  (aws/invoke client {:op :PutObject :request {:Bucket (:name bucket) :Key key
                                               :Body bytes}}))

(defn read-s3
  "given a key returns a stream
 
   (component/with [storage (storage/create -conf-)]
     (slurp (read-s3 storage \"hello\")))
   => \"Hello World\""
  {:added "0.1"}
  [{:keys [bucket client]} key]
  (:Body (aws/invoke client {:op :GetObject :request {:Bucket (:name bucket) :Key key}})))

(defn list-s3
  "lists all keys in the storage
 
   (component/with [storage (storage/create -conf-)]
     (list-s3 storage))
   => [\"hello\" \"sample\"]"
  {:added "0.1"}
  [{:keys [bucket client]}]
  (map :Key (:Contents (aws/invoke client {:op :ListObjects :request {:Bucket  (:name bucket)}}))))

(defn info-s3
  "gets the info
 
   (component/with [storage (storage/create -conf-)]
     (info-s3 storage \"hello\"))
   => {:size 11}"
  {:added "0.1"}
  [{:keys [bucket client]} key]
  (if-let [size (:ContentLength (aws/invoke client {:op :HeadObject :request {:Bucket (:name bucket) :Key key}}))]
    {:size size}))

(defn clear-s3
  "clears either the entire store or a single key
   
   (component/with [storage (storage/create -conf-)]
     (clear-s3 storage))
   => {:Deleted [{:Key \"hello\"} {:Key \"sample\"}]}
 
   (component/with [storage (storage/create -conf-)]
     (write-s3 storage \"hello\" (.getBytes \"Hello World\"))
     [(clear-s3 storage \"hello\")
      (list-s3 storage)])
   => [{} [\"sample\"]]"
  {:added "0.1"}
  ([{:keys [bucket client] :as storage}]
   (let [ks   (list-s3 storage)
         objs (mapv #(hash-map :Key %) ks)]
     (aws/invoke client {:op :DeleteObjects :request {:Delete {:Objects objs} :Bucket  (:name bucket)}})))
  ([{:keys [bucket client]} key]
   (aws/invoke client {:op :DeleteObject :request {:Bucket (:name bucket) :Key key}})))

(defn create-client
  "constructs a raw aws client
 
   (create-client {:endpoint    {:protocol :http
                                 :hostname \"127.0.0.1\"
                                 :port 9000}
                   :credentials {:access-key-id     \"AKIAIOSFODNN7EXAMPLE\"
                                 :secret-access-key \"wJalrXUtnFEMI/K7MDENG/bPxRfiCYEXAMPLEKEY\"}})
   => cognitect.aws.client.Client"
  {:added "0.1"}
  [{:keys [endpoint credentials] :as m}]
  (aws/client {:api :s3
               :region :us-east-1
               :credentials-provider (credentials/basic-credentials-provider credentials)
               :endpoint-override endpoint}))

(defn has-bucket?
  "checks if the bucket is available
 
   (component/with [storage (storage/create -conf-)]
     (has-bucket? storage))
   => true
 
   (component/with [storage (storage/create -conf-)]
     (has-bucket? (assoc-in storage [:bucket :name] \"another\")))
   => false"
  {:added "0.1"}
  ([{:keys [bucket client]}]
   (not (get (aws/invoke client {:op :HeadBucket :request {:Bucket (:name bucket)}}) :cognitect.anomalies/category))))

(defn create-bucket
  "creates a bucket on s3
   
   (component/with [storage (storage/create -conf-)]
     (create-bucket (assoc-in storage [:bucket :name] \"another\")))
   => {:Location \"/another\"}"
  {:added "0.1"}
  ([{:keys [bucket client]}]
   (aws/invoke client {:op :CreateBucket :request {:Bucket (:name bucket)}})))

(defn delete-bucket
  "deletes a bucket from s3
 
   (component/with [storage (storage/create -conf-)]
     (delete-bucket (assoc-in storage [:bucket :name] \"another\")))
   => {}"
  {:added "0.1"}
  ([{:keys [bucket client]}]
   (aws/invoke client {:op :DeleteBucket :request {:Bucket (:name bucket)}})))

(defrecord S3Storage [client]
  Object
  (toString [storage]
    (if (has-bucket? storage)
      (str {:items (count (list-s3 storage))})
      (str [:inactive])))
  
  protocol.storage/IStorage
  (-write    [storage key bytes] (write-s3 storage key bytes) storage)
  (-read     [storage key] (read-s3 storage key))
  (-list     [storage] (list-s3 storage))
  (-info     [storage key] (info-s3 storage key))
  (-clear    [storage] (clear-s3 storage) storage)
  (-clear    [storage key] (clear-s3 storage key) storage)
  
  protocol.component/IComponent
  (-start [{:keys [bucket initial] :as storage}]
    (cond (#{:temp :refresh} (:create bucket))
          (doto storage
            (clear-s3)
            (delete-bucket)
            (create-bucket)))
    (doseq [[k bytes] initial]
      (write-s3 storage k bytes))
    storage)
  (-stop  [{:keys [bucket] :as storage}]
    (cond (#{:temp} (:create bucket))
          (doto storage
            (clear-s3)
            (delete-bucket)))
    storage))

(defmethod print-method S3Storage
  ([v ^java.io.Writer w]
   (.write w (str #"storage.s3 " v))))

(defmethod protocol.storage/-create :s3
  ([m]
   (map->S3Storage (assoc m :client (create-client m)))))
