(ns hara.platform.mail
  (:require [hara.protocol.mail :as protocol.mail]
            [hara.platform.mail.local :deps true]
            [hara.platform.mail.mock :deps true]
            [hara.platform.mail.remote :deps true]
            [hara.core.component :as component]
            [hara.module :as module]))

(module/include (hara.platform.mail.interop.message message))

(defn send-mail
  "sends mail to the given address
 
   (-> (mailer {:type :mock})
       (send-mail {:to [\"a@a.com\"]
                  :body \"Test\"}))"
  {:added "3.0"}
  [mailer {:keys [to cc bcc subject body] :as message}]
  (protocol.mail/-send-mail mailer message))

(defn send-bulk
  "sends a list of mail to the given address
 
   (-> (mailer {:type :mock})
       (send-bulk [{:to [\"a@a.com\"]
                   :body \"Test A\"}
                   {:to [\"b@b.com\"]
                    :body \"Test B\"}
                   {:to [\"c@c.com\"]
                    :body \"Test C\"}]))"
  {:added "3.0"}
  [mailer messages]
  (protocol.mail/-send-bulk mailer messages))

(defn init-mail
  "sets the mailbox with the inital messages
   (let [box (mailbox {:id :test
                       :type :mock})]
     (init-mail box (mapv message [{:to [\"a@a.com\"]
                                    :body \"Test A\"}]))
     (count-mail box))
   => 1"
  {:added "3.0"}
  [mailbox messages]
  (protocol.mail/-init-mail mailbox messages))

(defn count-mail
  "counts the number of mail in the mailbox
 
   (let [box (mailbox {:id :test
                       :type :mock
                       :addresses [\"a@a.com\"]})]
     (-> (mailer {:type :mock})
         (send-bulk [{:to [\"a@a.com\"] :body \"Test A\"}
                     {:to [\"b@b.com\"] :body \"Test B\"}
                     {:to [\"c@c.com\"] :body \"Test C\"}]))
     (count-mail box))
  => 1"
  {:added "3.0"}
  [mailbox]
  (protocol.mail/-count-mail mailbox))

(defn get-mail
  "retrieves mail at a given index
 
   (let [box (mailbox {:id :test
                       :type :mock
                       :addresses [\"a@a.com\"]})]
     (-> (mailer {:type :mock})
         (send-mail {:to [\"a@a.com\"] :body \"Test A\"}))
     (-> (get-mail box 0)
         (object/to-data)))
   => (contains {:to [\"a@a.com\"]
                :body \"Test A\"})"
  {:added "3.0"}
  [mailbox i]
  (protocol.mail/-get-mail mailbox i))

(defn list-mail
  "retrieves all mail in the mailbox
   
   (let [box (mailbox {:id :test
                       :type :mock
                       :addresses [\"a@a.com\"]})]
     (-> (mailer {:type :mock})
         (send-bulk [{:to [\"a@a.com\"] :body \"Test A\"}
                     {:to [\"a@a.com\"] :body \"Test B\"}
                     {:to [\"a@a.com\"] :body \"Test C\"}]))
     (-> (list-mail box)
         (object/to-data)))
   => (contains-in [{:body \"Test A\"}
                   {:body \"Test B\"}
                    {:body \"Test C\"}])"
  {:added "3.0"}
  [mailbox]
  (protocol.mail/-list-mail mailbox))

(defn clear-mail
  "clears mail for the entire mailbox
 
   (let [box (mailbox {:id :test
                       :type :mock
                       :addresses [\"a@a.com\"]})]
     (-> (mailer {:type :mock})
         (send-bulk [{:to [\"a@a.com\"] :body \"Test A\"}
                     {:to [\"a@a.com\"] :body \"Test B\"}
                     {:to [\"a@a.com\"] :body \"Test C\"}]))
     (clear-mail box)
     (count-mail box))
   => 0"
  {:added "3.0"}
  [mailbox]
  (protocol.mail/-clear-mail mailbox))

(defn create-mailer
  "creates a mailer for use with hara.core.component
 
   (create-mailer {:type :mock})
 
   (create-mailer {:type :local})
 
   (create-mailer -test-smtp-)"
  {:added "3.0"}
  [m]
  (protocol.mail/-create-mailer m))

(defn create-mailbox
  "creates a mailbox for use with hara.core.component
 
   (create-mailbox {:type :mock
                    :addresses [\"a@a.com\"]})
 
   (create-mailbox -test-imap-)"
  {:added "3.0"}
  [m]
  (protocol.mail/-create-mailbox m))

(defn copy-mailbox
  "copies the contents of one mailbox to another
 
   (def boxA (mailbox -test-imap-))
   
   (def boxB (mailbox {:id :test
                       :type :mock}))
   
   (copy-mailbox boxA boxB)
   
   (= (mapv (comp :all-headers object/to-data) (list-mail boxA))
      (mapv (comp :all-headers object/to-data) (list-mail boxB)))
   => true"
  {:added "3.0"}
  [from to]
  (let [messages (list-mail from)]
    (init-mail to messages)))

(defn mailer
  "creates an active mailer
 
   (mailer {:type :mock})
   
   (mailer -test-smtp-)"
  {:added "3.0"}
  [m]
  (-> (create-mailer m)
      (component/start)))

(defn mailbox
  "creates an active mailbox
 
   (mailbox {:type :mock})
   ;; => #mailbox.mock{:id \"8c465d04-fdaa-4401-997e-cd6ed3d9d732\",
   ;;                  :messages 0, :addresses #{}}
 
   (mailbox -test-pop3-)"
  {:added "3.0"}
  [m]
  (-> (create-mailbox m)
      (component/start)))
