(ns hara.platform.mail.interop.body
  (:require [hara.core.base.encode :as encode]
            [hara.io.file :as fs]
            [hara.object :as object]
            [hara.security :as security]
            [hara.platform.mail.interop.wrap :as wrap])
  (:import [javax.mail.internet MimeBodyPart MimeMultipart]))

(defn part-empty
  "creates an empty MimeBodyPart object
 
   (part-empty)
   ;;=> #part{:size -1, :content-type \"text/plain\", :line-count -1}
   "
  {:added "3.0"}
  ([] (part-empty nil))
  ([_]
   (MimeBodyPart.)))

(def part-exclusions [:content-stream :raw-input-stream :data-handler
                      :input-stream :data-handler :all-header-lines :all-headers])

(def part-read-methods
  (-> (apply dissoc (object/read-getters MimeBodyPart) :disposition part-exclusions)
      (update-in [:content] wrap/wrap-suppress)
      (merge {:type (fn [p] (keyword (.getDisposition p)))})))

(def part-write-methods
  (-> (apply dissoc (object/write-setters MimeBodyPart) part-exclusions)
      (merge {:type {:type clojure.lang.Keyword
                     :fn (fn [p v] (doto p (.setDisposition (name v))))}})))

(defn digest
  "creates a digest for a given file
 
   (digest \"project.clj\")
   ;;=> \"70d941a9d512251bdc95180ccf2441f3\"
   "
  {:added "3.0"}
  ([path]
   (digest path "MD5"))
  ([path algo]
   (let [bytes (fs/read-all-bytes path)
         out   (security/digest bytes algo)]
     (encode/to-hex out))))

(defn set-file
  "creates a MimeBodyPart object based upon `:type`
 
   (-> (MimeBodyPart.)
       (set-file \"project.clj\")
       (object/to-data))
   => (contains {:type :attachment,
                 :file-name \"project.clj\",
                 :content-type \"text/plain\"})"
  {:added "3.0"}
  [part file]
  (let [f      (fs/file file)
        bytes  (fs/read-all-bytes file)
        md5    (digest file "MD5")]
    (doto part
      (.attachFile f)
      (.setContentMD5 md5))))

(object/map-like
 MimeBodyPart
 {:tag "part"
  :read    {:methods part-read-methods}
  :write   {:methods (assoc part-write-methods
                            :file {:type String
                                   :fn set-file})
            :empty part-empty}})

(defn part
  "creates a MimeBodyPart from scratch
 
   (part {:type :text :text \"Hello\"})
   ;;=> #part{:content \"Hello\", :type :text, :size -1,
   ;;         :content-type \"text/plain\", :line-count -1}
   "
  {:added "3.0"}
  [m]
  (object/from-data m MimeBodyPart))

(defn body-read
  "converts a MimeMultiPart object into a vector
 
   (-> (body [{:type :text :text \"Hello\"}])
       (body-read))
   => (contains-in [{:type :text
                     :content-type \"text/plain\"
                     :content \"Hello\"}])"
  {:added "3.0"}
  [b]
  (let [len (.getCount b)]
    (mapv (fn [i] (object/to-data (.getBodyPart b i)))
          (range len))))

(defn body
  "converts a vector into a MimeMultiPart object
   
   (body [{:type :attachment
           :description \"hello there\"
           :file \"project.clj\"}
          {:type :text
          :content-id \"abc\"
           :content-language [\"en\"]
           :text \"Hello\"}])"
  {:added "3.0"}
  [arr]
  (MimeMultipart. (into-array (map part arr))))

(object/vector-like
 MimeMultipart
 {:tag "body"
  :read  body-read
  :write body})
