(ns hara.test.form.listener
  (:require [hara.event :as event]
            [hara.test.common :as common]
            [hara.test.form.print :as print]))

(defn summarise-verify
  "extract the comparison into a valid format "
  {:added "3.0"}
  [result]
  {:status    (if (and (= :success (-> result :status))
                     (= true (-> result :data)))
              :success
              :failed)
   :path    (-> result :meta :path)
   :name    (-> result :meta :refer)
   :ns      (-> result :meta :ns)
   :line    (-> result :meta :line)
   :desc    (-> result :meta :desc)
   :form    (-> result :actual :form)
   :check   (-> result :checker :form)
   :actual  (-> result :actual :data)})

(defn summarise-evaluate
  "extract the form into a valid format"
  {:added "3.0"}
  [result]
  {:status  (-> result :status)
   :path    (-> result :meta :path)
   :name    (-> result :meta :refer)
   :ns      (-> result :meta :ns)
   :line    (-> result :meta :line)
   :desc    (-> result :meta :desc)
   :form    (-> result :form)
   :actual  (-> result :data)})

(defn form-printer
  "prints out result for each form"
  {:added "3.0"}
  [{:keys [result]}]
  (when (and (-> result :status (= :exception))
           (common/*print* :print-thrown))
    (.beep (java.awt.Toolkit/getDefaultToolkit))
    (print/print-thrown (summarise-evaluate result))))

(defn check-printer
  "prints out result per check"
  {:added "3.0"}
  [{:keys [result]}]
  (when (or (and (-> result :status (= :exception))
               (common/*print* :print-failure))
          (and (-> result :data (= false))
               (common/*print* :print-failure)))
    (.beep (java.awt.Toolkit/getDefaultToolkit))
    (print/print-failure (summarise-verify result)))
  (if (and (-> result :data (= true))
           (common/*print* :print-success))
    (print/print-success (summarise-verify result))))

(defn form-error-accumulator
  "accumulator for thrown errors"
  {:added "3.0"}
  [{:keys [result]}]
  (when common/*errors*
    (if (-> result :status (= :exception))
      (swap! common/*errors* update-in [:exception] conj result))))

(defn check-error-accumulator
  "accumulator for errors on checks"
  {:added "3.0"}
  [{:keys [result]}]
  (when common/*errors*
    (if (or (-> result :status (= :exception))
            (-> result :data (= false)))
      (swap! common/*errors* update-in [:failed] conj result))))

(defn fact-printer
  "prints out results after every fact"
  {:added "3.0"}
  [{:keys [meta results skipped]}]
  (if (and (common/*print* :print-facts)
           (not skipped))
    (print/print-fact meta results)))

(defn fact-accumulator
  "accumulator for fact results"
  {:added "3.0"}
  [{:keys [id meta results]}]
  (reset! common/*accumulator* {:id id :meta meta :results results}))

(defn bulk-printer
  "prints out the end summary"
  {:added "3.0"}
  [{:keys [results]}]
  (if (common/*print* :print-bulk)
    (print/print-summary results))
  (when common/*errors*
    (println "-------------------------")
    (when-let [failed (:failed @common/*errors*)]
      (doseq [result failed]
        (print/print-failure (summarise-verify result))))
    (when-let [exceptions (:exception @common/*errors*)]
      (doseq [result exceptions]
        (print/print-thrown (summarise-evaluate result))))
    (println "")))

(defn install-listeners
  "installs all listeners"
  {:added "3.0"}
  []
  (do (event/install-listener :test/form-printer  {:test :form}  form-printer)
      (event/install-listener :test/check-printer {:test :check} check-printer)
      (event/install-listener :test/form-error-accumulator {:test :form} form-error-accumulator)
      (event/install-listener :test/check-error-accumulator {:test :check} check-error-accumulator)
      (event/install-listener :test/fact-printer {:test :fact} fact-printer)
      (event/install-listener :test/fact-accumulator {:test :fact} fact-accumulator)
      (event/install-listener :test/bulk-printer {:test :bulk} bulk-printer)
      true))
