(ns hara.test.common
  (:require [clojure.main :as main]
            [hara.string :as string]
            [hara.core.base.result :as result]))

(defonce ^:dynamic *settings* {:test-paths ["test"]})

(defonce ^:dynamic *meta* nil)

(defonce ^:dynamic *desc* nil)

(defonce ^:dynamic *path* nil)

(defonce ^:dynamic *id* nil)

(defonce ^:dynamic *accumulator* (atom nil))

(defonce ^:dynamic *errors* nil)

(defonce ^:dynamic *print* #{:print-thrown :print-failure :print-bulk})

(defonce ^:dynamic *test-suffix* "-test")

(defonce ^:dynamic *root* nil)

(defrecord Op []
  Object
  (toString [op]
    (str "#op." (name (:type op)) (dissoc (into {} op) :type))))

(defmethod print-method Op
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn op
  "creates an 'op' for evaluation
 
   (op {:type :form :form '(+ 1 1)})
   => hara.test.common.Op"
  {:added "3.0"}
  [m]
  (map->Op m))

(defn op?
  "checks to see if a datastructure is an 'Op'
 
   (op? (op {:type :form :form '(+ 1 1)}))
   => true"
  {:added "3.0"}
  [x]
  (instance? Op x))


(defn function-string
  "returns the string representation of a function
 
   (function-string every?) => \"every?\"
 
   (function-string reset!) => \"reset!\""
  {:added "3.0"}
  [func]
  (-> (type func)
      str
      (string/split #"\$")
      last
      main/demunge))

(defrecord Checker [fn]
  Object
  (toString [{:keys [expect tag]}]
    (str "#" (name tag) (cond (coll? expect)
                              expect

                              (fn? expect)
                              (str "<" (function-string expect) ">")

                              :else
                              (str "<" expect ">"))))

  clojure.lang.IFn
  (invoke [ck data] (let [func (:fn ck)] (func data))))

(defmethod print-method Checker
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn checker
  "creates a 'hara.test.common.Checker' object
 
   (checker {:tag :anything :fn (fn [x] true)})
   => hara.test.common.Checker"
  {:added "3.0"}
  [m]
  (map->Checker m))

(defn checker?
  "checks to see if a datastructure is a 'hara.test.common.Checker'
 
   (checker? (checker {:tag :anything :fn (fn [x] true)}))
   => true"
  {:added "3.0"}
  [x]
  (instance? Checker x))

(defn evaluate
  "converts a form to a result
 
   (->> (evaluate {:form '(+ 1 2 3)})
        (into {}))
   => (contains {:status :success, :data 6, :form '(+ 1 2 3), :from :evaluate})"
  {:added "3.0"}
  [{:keys [form]}]
  (let [out (try
              {:status :success :data (eval form)}
              (catch Throwable t
                {:status :exception :data t}))]
    (result/result (assoc out :form form :from :evaluate))))
