(ns hara.time.joda.data
  (:require [hara.protocol.string :as protocol.string]
            [hara.protocol.time :as protocol.time]
            [hara.time.data.coerce :as coerce])
  (:import (org.joda.time DateTime DateTimeZone)
           (org.joda.time.format DateTimeFormatter)))

(defmethod protocol.time/-time-meta DateTimeZone
  [_]
  {:base :timezone})

(extend-type DateTimeZone
  protocol.string/IString
  (protocol.string/-to-string [tz]
    (.getID tz)))

(defmethod protocol.string/-from-string DateTimeZone
  [s _ _]
  (DateTimeZone/forID s))

(defn from-map
  "creates a ZonedDateTime object from a map
 
   (joda/from-map  (t/epoch {:timezone \"GMT\"}))
   => org.joda.time.DateTime"
  {:added "3.0"}
  [{:keys [millisecond second minute hour day month year timezone]}]
  (DateTime. ^int year ^int month ^int day ^int hour ^int minute ^int second ^int millisecond
             ^DateTimeZone (coerce/coerce-zone timezone {:type DateTimeZone})))

(defmethod protocol.time/-time-meta DateTime
  [_]
  {:base :instant
   :formatter {:type DateTimeFormatter}
   :parser    {:type DateTimeFormatter}
   :map {:from  {:fn from-map}}})

(extend-type DateTime
  protocol.time/IInstant
  (-to-long       [t] (.getMillis t))
  (-has-timezone? [t] true)
  (-get-timezone  [t] (protocol.string/-to-string (.getZone t)))
  (-with-timezone [t tz]
    (.withZone
     t
     ^DateTimeZone (coerce/coerce-zone tz {:type DateTimeZone})))

  protocol.time/IRepresentation
  (-millisecond  [t _] (.getMillisOfSecond t))
  (-second       [t _] (.getSecondOfMinute t))
  (-minute       [t _] (.getMinuteOfHour t))
  (-hour         [t _] (.getHourOfDay t))
  (-day          [t _] (.getDayOfMonth t))
  (-day-of-week  [t _] (.getDayOfWeek t))
  (-month        [t _] (.getMonthOfYear t))
  (-year         [t _] (.getYear t)))

(defmethod protocol.time/-from-long DateTime
  [^Long long {:keys [timezone]}]
  (DateTime. long ^DateTimeZone (coerce/coerce-zone timezone {:type DateTimeZone})))

(defmethod protocol.time/-now DateTime
  [{:keys [timezone]}]
  (.withZone (DateTime.)
             (coerce/coerce-zone timezone {:type DateTimeZone})))
