(ns hara.tool.monitor
  (:require [hara.object.framework.read :as read]
            [hara.data.base.map :as map]
            [hara.watch :as watch]
            [hara.object.framework.struct :as struct]
            [hara.function.base.invoke :refer [definvoke]]
            [hara.function.procedure :as procedure]
            [hara.function.procedure.registry :as registry]))

(defonce ^:dynamic *registry* (atom {}))

(defrecord Monitor [state opts]
  clojure.lang.IDeref
  (deref [monitor] @state)

  Object
  (toString [monitor]
    (str "#monitor" @state)))

(defmethod print-method Monitor
  [v w]
  (.write w (str v)))

(defn running-monitor?
  "checks if a monitor is running
 
   (def viewer (monitor \"\" {:start false}))
   
   (running-monitor? viewer)
   => false"
  {:added "3.0"}
  [monitor]
  (boolean (if-let [instance (:instance @(:opts monitor))]
             (procedure/procedure-running? instance))))

(definvoke start-monitor-loop
  "helper function for `start-monitor`"
  {:added "3.0"}
  [:procedure {:name :hara.tool.monitor
               :registry *registry*
               :mode  :async
               :arglist [:monitor :instance :notify]}]
  ([{:keys [state opts] :as monitor} instance notify]
   (let [{:keys [object interval accessor]} @opts]
     (swap! opts assoc :instance instance)
     (deliver notify true)
     (loop []
       (reset! state (accessor object))
       (Thread/sleep interval)
       (recur)))))

(defn start-monitor
  "starts a monitor viewer
 
   (-> (start-monitor viewer)
       (running-monitor?))
   => true"
  {:added "3.0"}
  [monitor]
  (when-not (running-monitor? monitor)
    (let [notify (promise)]
      (start-monitor-loop monitor nil notify)
      @notify))
  monitor)

(defn getter-accessor
  "creates a function that returns object data
 
   ((getter-accessor {:bytes (fn [s] (.getBytes s))
                      :count count})
    \"hello\")
   => (contains {:bytes bytes?, :count 5})"
  {:added "3.0"}
  [getters]
  (fn [obj]
    (map/map-vals #(% obj) getters)))

(defn monitor
  "creates a monitor viewer
 
   (monitor \"\" {:start false})"
  {:added "3.0"}
  ([obj]
   (monitor obj {}))
  ([obj {:keys [start getters struct access interval watches]
         :or {start true
              interval 50
              watches []
              access :getter}}]
   (let [accessor (cond struct
                        (struct/struct-accessor struct access)
                        
                        :else
                        (getter-accessor (or getters
                                             (-> (type obj) (read/meta-read) :methods))))
         state (atom (accessor obj))
         _     (doseq [[k f] watches]
                 (watch/add state k
                            (fn [_ _ _ n] (f n))
                            {:diff true}))
         monitor (->Monitor state
                      (atom {:object obj
                             :interval interval
                             :access   access
                             :accessor accessor
                             :struct   struct
                             :getters  getters
                             :watches  watches}))
         _    (if start (start-monitor monitor))]
     monitor)))

(defn monitor?
  "check for instance of `hara.tool.monitor.Monitor`
 
   (monitor? (monitor \"\" {:start false}))
   => true"
  {:added "3.0"}
  [obj]
  (instance? Monitor obj))

(defn stop-monitor
  "stops a monitor viewer
 
   (-> (stop-monitor viewer)
       (running-monitor?))
   => false"
  {:added "3.0"}
  [monitor]
  (swap! (:opts monitor)
         (fn [{:keys [instance] :as m}]
           (if instance (procedure/procedure-kill instance))
           (dissoc m :instance)))
  monitor)

(defn restart-monitor
  "restarts a monitor viewer
 
   (-> (restart-monitor viewer)
       (running-monitor?))
   false"
  {:added "3.0"}
  [monitor]
  (doto monitor
    (stop-monitor)
    (start-monitor)))

(defn live-monitors
  "lists all running viewers
 
   (mapv stop-monitor (live-monitors))"
  {:added "3.0"}
  []
  (mapv :monitor (registry/list-instances *registry* :hara.tool.monitor)))
