(ns <<project-ns>>.routes.guestbook
  (:require
   [clj-http.client :as http]
   [taoensso.carmine :as car :refer (wcar)]
   [<<project-ns>>.common.result :refer [ok sorry]]
   [<<project-ns>>.db.redis :as redis]
   [<<project-ns>>.db.core :as db]
   [ring.middleware.cookies :as cookies]
   [reitit.ring.middleware.multipart :as multipart]
   [clojure.tools.logging :as log]
   [clojure.java.io :as io]
   [clojure.spec.alpha :as s]
   [spec-tools.core :as st]
   ))

(s/def ::first-name
  (st/spec
   {:spec            string?
    :swagger/default "redcreation"
    :reason          "名字必填"}))

(s/def ::last-name
  (st/spec
   {:spec            string?
    :swagger/default "hc"
    :reason          "姓不能为空"}))

(s/def ::email
  (st/spec
   {:spec            string?
    :swagger/default "demo@redcreation.net"
    :reason          "邮箱是必填项"}))

(s/def ::pass
  (st/spec
   {:spec            string?
    :swagger/default "demo"
    :reason          "密码不能为空"}))

(defn guestbook-routes []
  ["/demo"
   {:swagger {:tag ["guestbook-demo-api"]}}

   [""
    {:get  {:summary     "根据id获取guestbook--并设置cookies"
            :description "查询某id的guestbook数据"
            :parameters  {:query {:id int?}}
            :handler     (fn [{{{:keys [id]} :query} :parameters}]
                           {:status  200
                            :body    {:data (db/get-user {:id id})}
                            :cookies {:test-cookie "123123123"}})}

     }]
   ["/http/get"
    {:get {:summary "一个获取httpclient调用的例子"
           :handler (fn [_]
                      (ok (:body
                           (http/get "http://cdn.imgs.3vyd.com/xh/admin/test.json" {:as :json}))))}}]

   ["/http/post/json"
    {:get {:summary "以json body提交"
           :handler (fn [_]
                      (ok (:body
                           (http/post "http://localhost:8185/management/public/doctor"
                                      {:form-params
                                       {:mobile   "15092107093",
                                        :nickName "marvin.ma",
                                        :name     "marvin",
                                        :openid   "8"}
                                       :content-type :json}))))}}]
   ["/http/post/form"
    {:get {:summary "以form body提交"
           :handler (fn [_]
                      (ok (:body
                           (http/post "http://localhost:8185/management/oauth2/token"
                                      {:form-params
                                       {:username   "test",
                                        :password   "test123",
                                        :client_id  "management-Client",
                                        :grant_type "password"}}))))}}]

   ["/redis/set"
    {:get {:summary    "set redis by key"
           :parameters {:query {:key   string?
                                :value string?}}
           :handler    (fn [{{{:keys [key, value]} :query} :parameters}]
                         {:status 200
                          :body   (redis/wcar*
                                   (car/ping
                                    (car/set key value)))})}}]

   ["/redis/get"
    {:get {:summary    "set redis by key"
           :parameters {:query {:key string?}}
           :handler    (fn [{{{:keys [key]} :query} :parameters}]
                         {:status 200
                          :body   {:data {key (redis/wcar*
                                               (car/get key))}}})}}]

   ;;api返回结果: {"data": "path params: {:id \"1\"}\n query params: {\"name\" \"2\"}\n body params: {:message \"22\"}"}
   ["/path/bad/:id"
    {:post {:summary    "路径上传参--不推荐此方法获取--query参数key变成了str"
            :parameters {:path  {:id int?}
                         :query {:name string?}
                         :body  {:message string?}}
            :handler    (fn [{:keys [path-params query-params body-params]}]
                          {:status 200
                           :body   {:data (str "path params: " path-params
                                               "\n query params: " query-params
                                               "\n body params: " body-params)}})}}]

   ;;api返回结果:
                                        ;{
                                        ;  "code": 1,
                                        ;  "message": "操作成功",
                                        ;  "data": "path params: {:id 1},  query params: {:name \"2\"},  body params: {:message \"22\"} "
                                        ;}
   ["/path/good/:id"
    {:post {:summary    "路径上传参--GOOD--获取到3种map"
            :parameters {:path  {:id int?}
                         :query {:name string?}
                         :body  {:message string?}}
            :handler    (fn [{{:keys [body query path]} :parameters}]
                          (ok (format "path params: %s,  query params: %s,  body params: %s " path query body)))}}]
   ;;api返回结果:
                                        ;{
                                        ;"code": 1,
                                        ;"message": "操作成功",
                                        ;"data": "path params 'id': 1, query params 'name': 2 , body params: {:message \"22\"} "
                                        ;}
   ["/path/good-all-params/:id"
    {:post {:summary    "路径上传参--GOOD--直接得到key的val"
            :parameters {:path  {:id int?}
                         :query {:name string?}
                         :body  {:message string?}}
            :handler    (fn [{{:keys [body]}          :parameters
                             {{:keys [id]} :path}    :parameters
                             {{:keys [name]} :query} :parameters}]
                          (ok (format "path params 'id': %s, query params 'name': %s , body params: %s " id name body)))}}]

   ["/ping"
    {:get (constantly (ok {:message "pong"}))}]

   ["/files"
    {:swagger {:tags ["demo-file"]}}

    ["/upload"
     {:post {:summary    "upload a file"
             :parameters {:multipart {:file multipart/temp-file-part}}
             :responses  {200 {:body {:name string?, :size int?}}}
             :handler    (fn [{{{:keys [file]} :multipart} :parameters}]
                           (ok {:name (:filename file)
                                :size (:size file)}))}}]

    ["/download"
     {:get {:summary "downloads a file"
            :swagger {:produces ["image/png"]}
            :handler (fn [_]
                       {:status  200
                        :headers {"Content-Type" "image/png"}
                        :body    (-> "public/img/warning_clojure.png"
                                     (io/resource)
                                     (io/input-stream))})}}]]

   ])
