(ns gram-api.hl
  (:require [gram-api.ll :as low-level]
            [clojure.tools.logging :as log]
            [clojure.data.json :as json]
            [clojure.string :as str])
  (:import (java.util.concurrent LinkedBlockingDeque BlockingDeque)))

(defn- shortify
  "Shortens string to 50 symbols and removes newlines"
  [s]
  (str/replace
    (->> (count s)
         (min 50)
         (subs s 0))
    \newline \space))

(defn try-read
  "Tries to parse json document from string. If it fails it returns nil and logs error message "
  [json]
  (try (json/read-str json)
       (catch Exception e
         (log/warnf "Failed to parse json: %s" (shortify json))
         (log/debug e))))


(defn next-update
  "Tries to get next message 10 times with delay 5000 ms. If it fails, it returns last error"
  [{:keys [api-key limit offset timeout] :or {limit 1 offset 1 timeout 10000} :as all}]
  (loop [attempt 10 last-error nil]
    (if (neg? attempt)
      (assoc all :error last-error)
      (let [response (low-level/get-updates api-key {:timeout timeout :limit limit :offset offset})]
        (if-let [json-body (try-read (:body response))]
          (if (get json-body "ok")
            (if-let [last-offset (get-in json-body ["result" 0 "update_id"])]
              (assoc all :offset (inc last-offset) :body json-body)
              (recur attempt nil))
            (if (= (get json-body "error_code") 401)
              (assoc all :error "Unauthorised. API key is incorrect")
              (recur (dec attempt) (get json-body "description"))))
          (do (Thread/sleep 5000)
              (recur (dec attempt) "Cannot parse result. Make sure API key is correct")))))))

(defn send-message-cycled
  "Tries to send message to chat_id. If it fails 10 times, it'll return last error.
  Cannot detect 429 error. Use enqueue-message instead"
  [{:keys [api-key chat_id text parse_mode] :or {parse_mode "none"} :as all}]
  (loop [attempt 10 last-error nil]
    (if (neg? attempt)
      (assoc all :error last-error)
      (let [response (low-level/send-message api-key {:chat_id    chat_id :text text
                                                      :parse_mode parse_mode})
            body (:body response)
            json-body (try-read body)
            error-code (get json-body "error_code")
            error-desc (get json-body "description")]
        (if (and json-body (not error-code))
          (assoc all :body json-body)
          (do (log/warnf "send-message-cycled: Error code: %s. %s" error-code error-desc)
              (Thread/sleep 5000)
              (recur (dec attempt) (format "send-message-cycled: Error code: %s. %s"
                                           error-code error-desc))))))))

(defn- parse-int-
  "Extracts first number from given string. If it fails, return 0"
  [s]
  (Integer/valueOf ^String (or (^String re-find #"\d+" (or s "0")) "0")))

(defn- wise-return [^BlockingDeque deque to-return]
  (loop [to-return to-return]
    (let [pending (butlast to-return)
          now (last to-return)]
      (if now (do (.offerFirst deque now)
                  (recur pending))))))

(defn- wise-take [^BlockingDeque deque]
  (let [size (.size deque)
        we-take (max (min size 5) 1)]
    (loop [acc-result [] taken-len 0]
      (if (<= we-take (count acc-result))
        acc-result
        (let [next (.takeFirst deque)]
          (if (-> (count next) (+ taken-len) (> 4095))
            (do (wise-return deque [next])
                acc-result)
            (recur (conj acc-result next) (+ taken-len (count next)))))))))

(defn- start-consuming [^BlockingDeque deque api-key chat_id]
  (future
    (loop []
      (let [unsent (wise-take deque)
            combined-unsent (str/join "\n" unsent)
            response (low-level/send-message api-key {:chat_id    chat_id :text combined-unsent
                                                      :parse_mode "Markdown"})
            body (:body response)]
        (if-let [json-body (try-read body)]
          (let [ok (get json-body "ok")
                error-code (get json-body "error_code")]
            (if ok
              (recur)
              (case (int error-code)
                429 (let [wait-ms (* (parse-int- (get json-body "description")) 1000)]
                      (log/warnf "queue-processor: Got error 429. I'll retry in %s ms" wait-ms)
                      (wise-return deque unsent)
                      (Thread/sleep wait-ms)
                      (recur))
                401 (log/error "queue-processor: Error 401. Unauthorised. Queue processing STOP")
                (log/errorf "queue-processor: Error code: %s. Description: %s" error-code
                            (get json-body "description")))))
          (do (log/warnf "queue-processor: Server seems to be unresponsible, retrying in 5s.")
              (wise-return deque unsent)
              (Thread/sleep 5000)
              (recur)))))))

(defn- ^BlockingDeque prepare-queue-for-
  "Creates new message queue for given api-key and chat_id"
  [api-key chat_id]
  (let [deque (LinkedBlockingDeque.)]
    (start-consuming deque api-key chat_id)
    deque))

(defn enqueue-message
  "Puts message to queue. If queue is nil, it will be created for given api-key and chat_id.
  Returns input map with :queue replaced if it was required."
  [{:keys [api-key chat_id text queue] :as all}]
  (let [^BlockingDeque queue (or queue (prepare-queue-for- api-key chat_id))]
    (.addAll queue (->> (partition 4095 4095 [""] text) (map str/join)))
    (assoc all :queue queue)))

(defn escape-markdown [text]
  (str/replace text #"[\\*\\_\[`]" #(str "\\" %)))

(defn get-file-url
  "For given file prepares and returns download url on telegram server"
  [{:keys [api-key file_id] :as all}]
  (let [response (low-level/get-file api-key {:file_id file_id})
        body (:body response)
        json-body (try-read body)
        ok (get json-body "ok")
        description (get json-body "description")
        path (get-in json-body ["result" "file_path"])]
    (if ok (assoc all :file_path (str low-level/server "file/bot" api-key "/" path))
           (assoc all :error description))))
