# gram-api

A Clojure library that proxies function calls to telegram http long polling API methods
Currently two methods supported: getUpdates and sendMessage

It also contains high level functions designed to simplify common tasks
These functions are: next-update, send-message-cycled, enqueue-message

## Installation

`gram-api` is available as a Maven artifact
from [Clojars](https://clojars.org/hdghg/gram-api):

```clojure
[hdghg/gram-api "0.4.0"]
```


## Usage

Low-level functionality provided by namespace `gram-api.ll`

```clojure
(require '[gram-api.ll :as low-level])
```

This namespace supports two functions get-updates and send-message
(as reflections to methods in [Telegram bot API](https://core.telegram.org/bots/api)):

```clojure
(low-level/send-message "secret:api-key" {:chat_id 1337 :text "Hi there!"})
=>
{:headers {"server" "nginx/1.10.0", ..."},
 :status 200,
 :body "{\"ok\":true, ...}"}

(low-level/get-updates "secret:api-key" {:limit 1 :offset 1 :timeout 10000})
=>
{:headers {"server" "nginx/1.10.0", ..."},
 :status 200,
 :body "{\"ok\":true, ...}"}

(low-level/get-file "secret:api-key" {:file_id "1337"})
=>
{:headers {"server" "nginx/1.10.0", ..."},
 :status 200,
 :body "{\"ok\":true, ...}"}
```


High-level functionality is much more complicated and accessed in namespace `gram-api.hl`

```clojure
(require '[gram-api.hl :as high-level])
```

This namespace supports three functions next-update send-message-cycled and enqueue-message:

```clojure
(high-level/next-update {:api-key "secret:api-key"})
=>
{:api-key "secret:api-key",
 :limit 1,
 :offset 31337,
 :timeout 100000,
 :body {"ok" true, ...}}
```
Here, offset is an automatically incremented returned offset value.
Result may be retrieved using `(:body next-update-result)`
If any error occurred, next-update will log it, wait 5 seconds and then retry.
Thread will be blocked during that time. If after 10 times there is still a error,
last error will be associated into input parameter and returned.
It may be retrieved using `(:error next-update-result)`


```clojure
(high-level/send-message-cycled {:api-key "secret:api-key" :chat_id    1337
                                 :text    "Hello again"    :parse_mode "none" })
=>
{:api-key "secret:api-key",
 :chat_id 1337,
 :text "Hello again!",
 :body {"ok" true, ...}}
```
Result may be retrieved using `(:body send-message-cycled-result)`
If any error occurred, send-message-cycled will log it, wait 5 seconds and then retry.
Thread will be blocked during that time. If after 10 times there is still a error,
last error will be associated into input parameter and returned.
It may be retrieved using `(:error send-message-cycled-result)`


```clojure
(high-level/enqueue-message {:api-key "secret:api-key" :chat_id    1337
                            :text    "Hello again"})
=>
{:api-key "secret:api-key",
 :chat_id -178806972,
 :text "123",
 :queue #object[clojure.lang.Atom ...]}
```
This is most interesting function as it only send message immediately if queue is not in
'waiting' state. Queue may become 'waiting' if server responded error. In particular,
server may response error 429 if you send messages too fast.
(At time this being written if more then 40 messages in 5 minute interval)
If queue is not in waiting state, message will be sent synchronously. If it waits,
then message will be conj-ed to inner queue and send asynchronously as soon as it's
inner smart timer mechanism decides to retry.
Asynchronous messages always stacked into groups of maximum 10 messages in 1 message to
lower quantity of calls to telegram API.
You may want to escape markdown as enqueue-message always sends messages with
`:parse_mode "Markdown"` parameter. Use `(high-level/escape-markdown text)`

```clojure
(high-level/get-file-url {:api-key "secret:api-key" :file_id "1337"})
=> "https://api.telegram.org/file/botsecret:api-key/photo/file_3.jpg"
```

## License

Released under the MIT License:
<http://www.opensource.org/licenses/mit-license.php>
