# test.calculus

Generative integration tests using test.check. Tries to simplify the process of setting up and persisting test data.

## Rationale

Generative testing is a powerful tool, and one I would love to use as part of integration tests that exercise the full application stack.

One of the painful things about those kinds of (integration) tests, though, is the significant setup required. Most of the time you are not trying to generate a bunch of distinct, independent values, but rather a set of related records, which share foreign keys and/or join tables. There may be certain fields that must be unique across records. They'll also usually need to be inserted into the DB in a particular order to satisfy their table constraints. And often, certain fields (such as auto-incrementing id fields) are autogenerated by the database, and you'll need to shadow any existing bindings for that record to make sure your tests have access to these values. This can lead to a confusing mix of generation of values, setting up of relationships, and re-`let`ing records.

You can end up with integration tests that look like this:


```clojure

(deftest forum-posts-test
  (chuck/checking "get-posts" (chuck/times 10)

    [[dinesh aamira charlotte shonda hideo]
     (gen/vector-distinct-by :user_id (sgen/generator User) {:num-values 5})

     [funny-cat-gifs politics clojure-enthusiasts]
     (gen/vector-distinct-by :forum_id (sgen/generator Forum) {:num-values 3})

     [my-cat resisting-trump macro-confusion cat-cafes parenthesis-overload]
     (gen/vector-distinct-by :post_id (sgen/generator Post) {:num-values 5})]

    (jdbc/with-db-transaction [db th/test-db]
      (jdbc/set-rollback-only! db)
      (jdbc/insert-multi! db :user [dinesh charlotte shonda])
      (jdbc/insert-multi! db :forum [funny-cat-gifs politics clojure-enthusiasts])
      (jdbc/insert-multi! db :membership [{:forum_id (:forum_id funny-cat-gifs) :user_id (:user_id dinesh)}
                                          {:forum_id (:forum_id funny-cat-gifs) :user_id (:user_id charlotte)}
                                          {:forum_id (:forum_id funny-cat-gifs) :user_id (:user_id shonda)}
                                          {:forum_id (:forum_id politics) :user_id (:user_id aamira)}
                                          {:forum_id (:forum_id politics) :user_id (:user_id shonda)}
                                          {:forum_id (:forum_id politics) :user_id (:user_id hideo)}
                                          {:forum_id (:forum_id clojure-enthusiasts):user_id (:user_id dinesh)}
                                          {:forum_id (:forum_id clojure-enthusiasts) :user_id (:user_id aamira)}])

      (let [my-cat               (assoc my-cat :user_id (:user_id charlotte) :forum_id (:forum_id funny-cat-gifs))
            cat-cafes            (assoc cat-cafes :user_id (:user_id shonda) :forum_id (:forum_id funny-cat-gifs))
            resisting-trump      (assoc resisting-trump :user_id (:user_id hideo) :forum_id (:forum_id politics))
            macro-confusion      (assoc macro-confusion :user_id (:user_id dinesh) :forum_id (:forum_id clojure-enthusiasts))
            parenthesis-overload (assoc parenthesis-overload :user_id (:user_id aamira) :forum_id (:forum_id clojure-enthusiasts))]
        (jdbc/insert-multi! db :post [my-cat cat-cafes resisting-trump macro-confusion parenthesis-overload])

        (is (= (sort-by :post_id [macro-confusion parenthesis-overload])
               (sort-by :post_id (h/get-posts (tc/test-db) clojure-enthusiasts))))))))
```

It's an exaggeration, because you can of course write helper functions to clean up some of the repeated code. But you end up with lots of little helper functions (`user-joins-forum`, `user-writes-post`) that can all work slightly differently, and may or may not make changes to the records they take as their arguments, meaning you need to rebind them in a shadowing `let` binding. It's not the worst thing in the world, but it does make the test setup harder to follow, make it harder to quickly read and understand _what_ is being set up, and harder to distinguish the setup from the actual test code.

The aims of this project are to:

1. Reduce the amount and complexity of code required to set up your generated test data
2. Make test setup more homogenous and declarative, so it is easier to scan
3. Make test setup more easily distinguished from the actual test code

I've made some progress towards these aims, but am still not totally convinced what I'm trying to do is a great idea, so have pushed this code in it's WIP state to get feedback. I usually run a mile from complex macros or "magic" syntax, and this is definitely both of those. But I think it does have the potential to clean up test code significantly.

Here's what a similar integration test to the one above would look like written with `test.calculus`:

```clojure
(tc/use-test-db th/test-db)

(deftest forum-posts-test
  (tc/integration-test "get-posts" 10

    ;; 'fixtures' are specified first. these are records that will be
    ;; generated from user defined generators and inserted into the DB before
    ;; the tests are run.
    [:users  [dinesh aamira charlotte shonda hideo]
     :forums [funny-cat-gifs politics clojure-enthusiasts]
     :posts  [my-cat resisting-trump macro-confusion cat-cafes parenthesis-overload]]

    ;; 'relationships' follow the fixtures. these trigger the use of
    ;; user-defined functions that set foreign keys/create the join table
    ;; records required to specify the given relationship in the db.
    [:member funny-cat-gifs      dinesh charlotte shonda]
    [:member politics            aamira shonda hideo]
    [:member clojure-enthusiasts dinesh aamira]

    [:author [charlotte funny-cat-gifs]      my-cat]
    [:author [shonda    funny-cat-gifs]      cat-cafes]
    [:author [hideo     politics]            resisting-trump]
    [:author [dinesh    clojure-enthusiasts] macro-confusion]
    [:author [aamira    clojure-enthusiasts] parenthesis-overload]

    ;; 'generators' - any other arbitrary generators that are needed can be
    ;; specified here. these are not really needed for the below test, but are
    ;; just there as an example.
    [x gen/s-pos-int
     y gen/s-pos-int]

    (is (> (+ x y) x))
    (is (> (+ x y) y))

    (is (= (sort-by :post_id [macro-confusion parenthesis-overload])
           (sort-by :post_id (h/get-posts (tc/test-db) clojure-enthusiasts))))))
```

The keywords in the fixtures section look up generators defined by the user - these can be arbitrary test.check generator.

The keywords in the relationships section look up simple functions defined by the user.

Those fixtures and relationships look like this:

```clojure

(def leaf-generators
  {s/Int  gen/s-pos-int
   s/Inst datetime})

(tc/fixture :users
  {:table      :user
   :unique-key :user_id
   :generator  (sgen/generator User leaf-generators)})

(tc/fixture :forums
  {:table      :forum
   :unique-key :forum_id
   :generator  (sgen/generator Forum leaf-generators)})

(tc/fixture :posts
  {:table      :post
   :unique-key :post_id
   :generator  (sgen/generator Post leaf-generators)})

(tc/one-to-many :author
  (fn [[user forum] post]
    (assoc post
           :user_id  (:user_id user)
           :forum_id (:forum_id forum))))

(tc/many-to-many :member
  {:table :membership}
  (fn [forum user]
    {:forum_id (:forum_id forum)
     :user_id  (:user_id user)}))
```

## Developing test.calculus

`test.calculus` uses `docker-compose` to provision a test database. Run `docker-compose up -d` (or `make up`) to start it in the background, then `lein repl`. The port of the dockerized mysql will be automatically injected into the project's `env` using `lein-docker-compose`.

## clojure.spec

I haven't tried to integrate this with `clojure.spec` as yet, as I'm waiting for the 1.9 release before moving away from Prismatic Schema. However, `test.calculus` accepts arbitrary `test.check` generators for its fixtures, so I'm hoping that it should work just as well with `spec` as it does with `schema`.

## Feedback

As I say, I'm keen to get feedback. Is this a terrible idea or worth pursuing? Are there any pitfalls I should be considering? Does the syntax/structure of the integration test setup sort of make sense, or is it hopelessly confusing? Any ideas on how to improve it?
