(ns daifu.diagnosis.indicator
  (:require [daifu.diagnosis.indicator.activate :as activate]
            [daifu.diagnosis.indicator.invoke :as invoke]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [hara.data.nested :as nested]))

(def +indicator-list+ "daifu/defaults/indicators.edn")

(def +indicator-defaults+ {:level :info})

(defrecord Indicator []
  clojure.lang.IFn
  (invoke [indicator x]
          (invoke/invoke-indicator indicator x))
  (invoke [indicator x y]
          (invoke/invoke-indicator indicator x y))
  (invoke [indicator x y z]
          (invoke/invoke-indicator indicator x y z)))

(defmethod print-method Indicator
  [v w]
  (.write w (str "#" (name (:type v)) [(name (:id v)) (activate/activation v)] " "
                 {:* (vec (keys (dissoc v :id :type :main)))})))

(defn indicator
  "when passed a map, returns an indicator
   (-> (indicator {:id :hello-world
                   :type :function
                   :source '(fn [zloc] (zip/sexpr zloc))})
       (indicator?))
   => true"
  {:added "0.2"}
  [m]
  (-> (map->Indicator (nested/merge-nested +indicator-defaults+ m))
      (assoc :main (atom nil))))

(defn indicator?
  "checks if object is in fact in indicator
   (indicator? {:id :hello-world
                :type :function
                :source '(fn [zloc] (zip/sexpr zloc))})
   => false"
  {:added "0.2"}
  [x]
  (instance? Indicator x))

(defn load-files
  "loads indicators to a map from a list of files
   (->> [\"char_per_line.indi\" \"token_count.indi\"]
        (map #(str \"resources/daifu/defaults/indicators/function/\" %))
        (load-files))
   => (contains {:char-per-line indicator?
                 :token-count indicator?})"
  {:added "0.2"}
  [files]
  (->> files (map (fn [f] (-> (slurp f)
                              (read-string)
                              indicator)))
       (map (juxt :id identity))
       (into {})))

(defn load-defaults
  "loads default indicators
   (load-defaults)
   => map?"
  {:added "0.2"}
  ([] (load-defaults (io/resource +indicator-list+)))
  ([list]
   (->> (slurp list)
        (read-string)
        (map (fn [path]
               (io/resource (str "daifu/defaults/indicators/" path))))
        (load-files))))

(defn load-directory
  "loads indicators under a particular directory
   (-> (load-directory \"resources/daifu/defaults/indicators/idiom\")
       (keys))
   => (contains [:arithmatic :collection :control :equality :sequence :string] :in-any-order :gaps-ok)"
  {:added "0.2"}
  [dir]
  (->> (io/file dir)
       (file-seq)
       (filter (fn [f] (.endsWith (str f) ".indi")))
       (load-files)))

(defn activate-all
  "activates the map of indicators
   (->> (load-defaults)
        (activate-all)
        (vals)
        (every? #(= :activated (activate/activation %))))
   => true"
  {:added "0.2"}
  [indicators]
  (let [indicators (reduce-kv (fn [out k v]
                                (assoc out k (activate/activate v)))
                              {}
                              indicators)]
    (reduce-kv (fn [out k v]
                 (let [inputs (:inputs v)]
                   (if (and (= :activated (activate/activation v))
                            (every? #(-> indicators
                                         (get %)
                                         (activate/activation)
                                         (= :activated)) inputs))
                     (assoc out k v)
                     out)))
               {}
               indicators)))
