(ns daifu.diagnosis
  (:require [daifu.diagnosis.indicator :as indicator]
            [daifu.diagnosis.result :as result]
            [daifu.diagnosis.summary :as summary]
            [daifu.diagnosis.target :as target]
            [rewrite-clj.zip :as zip]
            [jai.query :as query]
            [clojure.pprint :as pprint]
            [clojure.tools.logging :as log]
            [clojure.string :as string]))

(defn retrieve
  "retrieves the contents of files from path
   (->> (retrieve (io/file \".\") (target/target target/+default-target+))
        (map :path))
   => #(every? (fn [path]
                 (.endsWith path \".clj\")) %)"
  {:added "0.2"}
  [repo target]
  (let [paths (target/pick-files repo target)
        paths (if-let [checks (:filter target)]
                (filter (fn [path]
                          (some (fn [check] (.endsWith path check)) checks)) paths)
                paths)
        paths (map (partial hash-map :path) paths)]
    (->> paths
         (map (partial target/retrieve-file repo))
         (map #(assoc %1 :reader %2) paths)
         (filter :reader))))

(defn diagnose-form-ns
  "gets the namespace of the file
   (diagnose-form-ns (zip/of-string \"(ns example.world)\"))
   => 'example.world"
  {:added "0.2"}
  [zloc]
  (second (zip/sexpr (first (query/select zloc '[ns] {:walk :top})))))

(defn diagnose-form-metas
  "gets the line and column information of a form
   (diagnose-form-metas (zip/of-string \"(defn hello [])\"))
   => '{:line 1, :column 1, :name hello}"
  {:added "0.2"}
  [zloc]
  (let [{:keys [row col]} (meta (zip/node zloc))]
    {:line row
     :column col
     :name (-> zloc zip/sexpr second)}))

(defn diagnose-form-file
  "returns a set of results for a single file from an indicator
   (diagnose-form-file (indicator/indicator {:id :no-docstring
                                             :type :function
                                             :source (fn [zloc]
                                                       (-> (zip/sexpr zloc)
                                                           (nth 2)
                                                           string?
                                                           not))})
                      {:path \"src/daifu/diagnosis/target/files.clj\"
                        :reader (io/reader \"src/daifu/diagnosis/target/files.clj\")})
   => (contains [{:line 9,
                  :column 1,
                  :name 'file-seq-filter,
                  :value true,
                  :path \"src/daifu/diagnosis/target/files.clj\",
                  :ns 'daifu.diagnosis.target.files,
                  :indicator :no-docstring,
                  :level :info}
                 result/result?
                 result/result?])"
  {:added "0.2"}
  [indicator {:keys [reader path]}]
  (let [zloc    (zip/of-string (slurp reader))
        ns      (diagnose-form-ns zloc)
        funcs   (query/select zloc [(or (:pattern indicator) '(#{defn defn-} & _))] {:walk :top})
        metas   (mapv diagnose-form-metas funcs)
        results (mapv indicator funcs)]
    (mapv (fn [meta value]
            (->> {:value value :path path :ns ns :indicator (:id indicator) :level (:level indicator)}
                 (merge meta)
                 result/result))
          metas results)))

(defn diagnose-form-all
  "returns a set of results for a target and an indicator
   (diagnose-form-all (io/file \".\")
                      (indicator/indicator {:id :no-docstring
                                            :type :function
                                            :source (fn [zloc]
                                                      (-> (zip/sexpr zloc)
                                                          (nth 2)
                                                          string?
                                                         not))})
                      (target/target target/+default-target+))
   => (contains [result/result?] :gaps-ok)"
  {:added "0.2"}
  [repo indicator target]
  (let [files   (retrieve repo target)
        results (keep (partial diagnose-form-file indicator) files)]
    (vec (apply concat results))))

(defn convert-expressions
  "converts clojure forms to pretty printed string"
  {:added "0.2"}
  [{:keys [expr alt] :as m}]
  (let [func (fn [m] (string/trim-newline (with-out-str (pprint/pprint m))))]
    (-> m
        (update-in [:expr] func)
        (update-in [:alt]  func))))

(defn diagnose-idiom-message
  "creates the output message for the idiom indicators"
  {:added "0.2"}
  [actual suggested]
  (format "Idiom Suggestion:\n Actual:\n%s\n Suggested:\n%s\n" actual suggested))

(defn diagnose-idiom-all
  "returns a set of results for targets and the idiom indicators
   (diagnose-idiom-all (io/file \".\")
                       (->> (slurp \"resources/daifu/defaults/indicators/idiom/arithmatic.indi\")
                            (read-string)
                            (indicator/indicator))
                       (target/target target/+default-target+))
   => []"
  {:added "0.2"}
  [repo indicator target]
  (let [files   (retrieve repo target)
        safe-fn (fn [f]
                  (try (let [results (indicator (:reader f))
                             ns (diagnose-form-ns (zip/of-file (:path f)))]
                         (map (fn [result]
                                  (let [{:keys [expr alt line column]}
                                        (convert-expressions result)]
                                    (result/result {:ns ns
                                                    :indicator (:id indicator)
                                                    :level :warn
                                                    :path (:path f)
                                                    :line line
                                                    :column column
                                                    :msg (diagnose-idiom-message expr alt)
                                                    :value {:actual expr :suggested alt}})))
                              results))
                       (catch Throwable t
                         (log/error t
                                    (str "Exception occured using" indicator "on file"
                                         (:path f))))))]
    (vec (mapcat safe-fn files))))

(defn diagnose
  "main entry point for running diagnosis of indicators
   (diagnose (io/file \".\")
             (->> (slurp \"resources/daifu/defaults/indicators/idiom/arithmatic.indi\")
                  (read-string)
                  (indicator/indicator))
             (target/target target/+default-target+))
   => []"
  {:added "0.2"}
  [repo indicator target]
  (case (:type indicator)
    :project  (indicator repo (assoc (target/read-project repo target) :target target))
    :function (diagnose-form-all repo indicator target)
    :form     (diagnose-form-all repo indicator target)
    :idiom    (diagnose-idiom-all repo indicator target)))
