(ns mongrove.verify-dc-migration
  (:require
    [mongrove.core :as mc])
  (:import
    java.util.UUID))

;; 1 or more threads for writes : inserts
;; can we store every write we make into a local atom ?
;; {id : object ...}

;; 1 or more threads for reading : fetch-one and query ?
;; what should be read concerns ?
;; queries use id only which can be any one of the id we have
;; previously created
;; compare against locally stored object
;; if we don't get the object, this indicates that we tried to read before
;; the secondary got the write. So start a new thread, wait a while and try again
;; if we still dont get the object, this is an error

;; do we need multiple DBs and collections ?

;; Metrics to check
;; replication lag, read latencies for cross region reads,
;; write latencies when primary is in other region

(def local-objects (atom {}))


(defn init-connection
  [connection-info]
  (mc/connect :replica-set
              connection-info))


(defn create-object
  [db coll-name]
  ;; what should be write concerns ?
  (let [new-object {:id (str (UUID/randomUUID))
                    :name (str (UUID/randomUUID))
                    :rank (rand-int 1000)}]
    (mc/insert db coll-name new-object)
    (swap! local-objects assoc (:id new-object) new-object)))


(defn read-object
  [db coll-name id]
  (let [local-object (get @local-objects id)
        db-object (mc/fetch-one db coll-name {:id id} :db db)]
    (if (nil? db-object)
      ;; if we don't get the object, this indicates that we tried to read before
      ;; the secondary got the write. So start a new thread, wait a while and try again
      ;; if we still dont get the object, this is an error
      (future
        (Thread/sleep 2000)
        (when (not= local-object (mc/fetch-one db coll-name {:id id} :db db))
          (println "Error in reading object after wait : " local-object)))
      ;; compare against locally stored object
      (when (not= local-object db-object)
        (println "Error in reading object : " local-object)))))


(comment
  (def primary (init-connection [{:host "shiva.local" :port 27017}]))

  (def primary-db (mc/get-db primary (str "migration_test_" (UUID/randomUUID))))
  (def coll-name "test")

  (create-object primary-db coll-name)

  (read-object primary-db coll-name (first (shuffle (keys @local-objects))))

  )
