(ns spongeblob.azure
  (:require [clojure.string :as s]
            [spongeblob.protocols :as proto :refer [BlobStore]]
            [spongeblob.util :refer [process-map starts-with?]])
  (:import com.microsoft.azure.storage.StorageCredentialsSharedAccessSignature
           com.microsoft.azure.storage.StorageException
           [com.microsoft.azure.storage.blob BlobProperties CloudBlobClient CloudBlobContainer CloudBlockBlob]
           java.io.ByteArrayOutputStream
           java.net.URI))

(declare get-client blob-ref)

(defrecord AzureBlobStorage [sas container base-uri cdn-uri])

;;; Not extending inline because of `get`.
(extend-type AzureBlobStorage
  BlobStore
  (cdn-url [this key] (if-let [cdn-uri (:cdn-uri this)]
                        (str cdn-uri "/" key)
                        (proto/url this key)))

  (url [this key] (s/join "/" [(:base-uri this) (:container this) key]))

  (exists? [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this) (:base-uri this) (:container this) key)]
      (.exists blob)))

  (get-metadata [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this) (:base-uri this) (:container this) key)]
      (when (.exists blob)
        (.downloadAttributes blob)
        (let [^BlobProperties props (.getProperties ^CloudBlockBlob blob)
              metadata (into {} (.getMetadata blob))]
          (if-let [content-type (.getContentType props)]
            (assoc metadata "content-type" content-type)
            metadata)))))

  (get-stream [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this) (:base-uri this) (:container this) key)]
      (try
        (with-open [^ByteArrayOutputStream baos (ByteArrayOutputStream.)]
          (if (.exists blob)
            (do
              ;; download contents into BAOS
              (.download blob baos)
              ;; sync the metadata (!)
              (.downloadAttributes blob)
              (let [^BlobProperties props (.getProperties ^CloudBlockBlob blob)]
                {:object-stream baos
                 :content-type (.getContentType props)
                 :content-length (.getLength props)}))))
        (catch  StorageException _
          (throw (ex-info (str "Key " key " doesn't exist!")
                          {:container (:container this)
                           :base-uri (:base-uri this)}))))))

  (get [this key]
    (let [object-map (proto/get-stream this key)]
      (assoc (select-keys object-map [:content-type :content-length])
             :data (.toByteArray ^ByteArrayOutputStream (:object-stream object-map)))))

  (put [this key content-type meta bytes]
    (let [^CloudBlockBlob blob (blob-ref (:sas this)
                                         (:base-uri this)
                                         (:container this)
                                         key)
          props (.getProperties blob)
          ^java.util.Map metadata (-> meta
                                      ;; We can't have keys that don't
                                      ;; adhere to C# identifier
                                      ;; naming rules.
                                      ;;
                                      ;; Ref: Naming and Referencing
                                      ;; Containers, Blobs, and
                                      ;; Metadata
                                      ;; https://msdn.microsoft.com/en-us/library/azure/dd135715.aspx
                                      (process-map :key-fn str :val-fn str))]
      (.setContentType props (or content-type "application/octet-stream"))
      (when metadata
        (.setMetadata blob (java.util.HashMap. metadata)))
      ;; basic caching setup
      (.setCacheControl props "max-age=3600, must-revalidate")
      (.uploadFromByteArray blob bytes 0 (count bytes))
      (str (.getUri blob))))

  (delete [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this)
                                         (:base-uri this)
                                         (:container this)
                                         key)]
      (try (.delete blob)
           (catch StorageException ex
             (if (starts-with? (.getMessage ex) "This request is not authorized")
               (throw (ex-info (str "You dont have enough permissions for deleting " key)
                               {:container (:container this)
                                :base-uri (:base-uri this)}))
               (throw (ex-info (.getMessage ex)
                               {:container (:container this)
                                :base-uri (:base-uri this)}))))))))


(defn ^:private get-client*
  [sas base-uri]
  (CloudBlobClient.
   (URI. base-uri)
   (StorageCredentialsSharedAccessSignature. sas)))


(def ^{:doc "Create a CloudBlobClient obj given sas and base-uri."}
  get-client
  (memoize get-client*))


(defn ^:private blob-ref
  "Given sas, base-uri, container & key get a reference to the blob."
  [sas base-uri container key]
  (let [^CloudBlobClient client (get-client sas base-uri)
        cont (.getContainerReference client container)]
    (.getBlockBlobReference cont key)))
