(ns spongeblob.azure
  (:require [clojure.string :as s]
            [spongeblob.protocols :as proto :refer [BlobStore]])
  (:import [com.microsoft.azure.storage.blob BlobProperties CloudBlobClient CloudBlobContainer CloudBlockBlob]
           com.microsoft.azure.storage.StorageCredentialsSharedAccessSignature
           java.io.ByteArrayOutputStream
           java.net.URI))

(declare get-client blob-ref)

(defrecord AzureBlobStorage [sas container base-uri cdn-uri])

;;; Not extending inline because of `get`.
(extend-type AzureBlobStorage
  BlobStore
  (cdn-url [this key] (proto/url this key))

  (url [this key] (s/join "/" [(:base-uri this) (:container this) key]))

  (exists? [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this) (:base-uri this) (:container this) key)]
      (.exists blob)))
  
  (get [this key]
    (let [^CloudBlockBlob blob (blob-ref (:sas this) (:base-uri this) (:container this) key)]
      (with-open [^ByteArrayOutputStream baos (ByteArrayOutputStream.)]
        (if (.exists blob)
          (do
            ;; download contents into BAOS
            (.download blob baos)
            ;; sync the metadata (!)
            (.downloadAttributes blob)
            (let [^BlobProperties props (.getProperties ^CloudBlockBlob blob)]
              {:data (.toByteArray baos)
               :content-type (.getContentType props)
               :content-length (.getLength props)}))
          (throw (ex-info (str "Key " key " doesn't exist!")
                          {:container (:container this)
                           :base-uri (:base-uri this)}))))))

  (put [this key meta bytes]
    (let [^CloudBlockBlob blob (blob-ref (:sas this)
                                         (:base-uri this)
                                         (:container this)
                                         key)
          props (.getProperties blob)]
      (when-let [ctype (:content-type meta)]
        (.setContentType props ctype))
      ;; basic caching setup
      (.setCacheControl props "max-age=3600, must-revalidate")
      (.uploadFromByteArray blob bytes 0 (count bytes))
      (str (.getUri blob)))))


(defn ^:private get-client*
  [sas base-uri]
  (CloudBlobClient.
   (URI. base-uri)
   (StorageCredentialsSharedAccessSignature. sas)))


(def ^{:doc "Create a CloudBlobClient obj given sas and base-uri."} get-client
  (memoize get-client*))

(defn ^:private blob-ref [sas base-uri container key]
  "Given sas, base-uri, container & key get a reference to the blob."
  (let [^CloudBlobClient client (get-client sas base-uri)
        cont (.getContainerReference client container)]
    (.getBlockBlobReference cont key)))

