(ns spongeblob.s3
  (:require [clojure.java.io :as io]
            [spongeblob.protocols :as proto :refer [BlobStore]])
  (:import com.amazonaws.auth.BasicAWSCredentials
           com.amazonaws.services.s3.AmazonS3Client
           [com.amazonaws.services.s3.model AmazonS3Exception CannedAccessControlList ObjectMetadata PutObjectRequest S3Object]))

(declare s3-client make-put-request ->bytes)

(defrecord S3BlobStorage [access-key secret-key bucket cloudfront-uri])

(extend-type S3BlobStorage
  BlobStore
  (cdn-url [this key] (if-let [cf-uri (:cloudfront-uri this)]
                        (str cf-uri "/" key)
                        (proto/url this key)))

  (url [this key] (str "https://" (:bucket this) ".s3.amazonaws.com" "/" key))

  (exists? [this key]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))]
      (try (.getObjectMetadata client (:bucket this) key)
           true
           (catch AmazonS3Exception s3e
             (if (= (.getStatusCode s3e) 404)
               false
               (throw s3e))))))
  
  (get [this key]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))]
      (try
        (with-open [^S3Object res (.getObject client ^String (:bucket this) ^String key)]
          {:content-length (.getContentLength (.getObjectMetadata res))
           :content-type (.getContentType (.getObjectMetadata res))
           :data (->bytes (.getObjectContent res))})
        (catch AmazonS3Exception _
          (throw (ex-info (str "Key " key " doesn't exist!")
                          {:bucket (:bucket this)}))))))

  (put [this key meta bytes]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))
          pr (make-put-request (:bucket this) key meta bytes)]
      (.putObject client pr)
      (.getResourceUrl client (:bucket this) key))))


(defn ^:private s3-client*
  "Given access and secret keys, construct the client object."
  [access-key secret-key]
  (AmazonS3Client. (BasicAWSCredentials. access-key secret-key)))


(def ^{:doc "Create a AmazonS3Client obj given access & secret keys."} s3-client
  (memoize s3-client*))


(defn ^:private make-put-request
  "Construct a file upload request object."
  ^PutObjectRequest
  [bucket key meta ^bytes bytes]
  (let [^ObjectMetadata ometa
        (doto (ObjectMetadata.)
          (.setContentType (:content-type meta "application/octet-stream"))
          (.setCacheControl "max-age=3600, must-revalidate")
          (.setContentLength (count bytes)))]
    (.withCannedAcl (PutObjectRequest. bucket key (io/input-stream bytes) ometa)
                    CannedAccessControlList/PublicRead)))


(defn ^:private ->bytes
  "Read an InputStream into a ByteArray."
  [is]
  (with-open [out (java.io.ByteArrayOutputStream.)]
    (clojure.java.io/copy (io/input-stream is) out)
    (.toByteArray out)))
