(ns heroku.datum
 (:import java.text.SimpleDateFormat)
 (:use
   [clojure.string :as string :only [join split]]))

(def long-re
  #"^-?[0-9]{1,18}$")

(def double-re
  #"^-?[0-9]+\.[0-9]+$")

(def attrs-re
  #"([a-zA-Z0-9_]+)(=?)([a-zA-Z0-9\.\_\-\:\/]*)")

(def standard-re
  #"\.*(\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d[\-\+]\d\d:00) ([a-z]+)\[([a-zA-Z\.0-9]+)\]\: (.*)$")

(def attrs-format "[user_data]{%s}\n")

(defn- re-match? [re s]
  (let [m (re-matcher re s)]
    (.find m)))

(defn- coerce-val [v]
  (cond
    (re-match? long-re v)
      (Long/parseLong v)
    (re-match? double-re v)
      (Double/parseDouble v)
    (= "" v)
      nil
    (= "true" v)
      true
    (= "false" v)
      false
    :else
      v))

(defn- parse-message-attrs [msg]
  (let [m (re-matcher attrs-re msg)]
    (loop [a (java.util.HashMap.)]
      (if (.find m)
        (do
          (when-not (= "" (.group m 2))
            (.put a
              (keyword (.group m 1))
              (coerce-val (.group m 3)))
            )
          (recur a))
        a))))

(defn- parse-timestamp [s]
  (let [f (SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ssZ")]
    (.getTime (.parse f (string/replace s #":\d\d$" "00")))))

(defn- parse-long [s]
  (if s (Long/parseLong s)))

(defn- line-to-map [l]
  (let [m (re-matcher standard-re l)]
    (if (.find m)
      (merge
        {:event_type "visibility"
         :timestamp (parse-timestamp (.group m 1))
         :app (.group m 2)
         :process (.group m 3)}
        (parse-message-attrs (.group m 4))))))

(defn read-map [line]
  (line-to-map line))

(defn- map-to-line [m]
  (string/join
    (drop-last
      (reduce
        #(str %1 (name (first %2)) "=" (second %2) " ")
        ""
        m))))

(defn write-map [m stream]
  (binding [*out* stream]
    (println (map-to-line m))
    (flush)))


