(ns hitokotonushi.common
  (:require (clojure      [string :as string])
            (clojure.java [jdbc   :as jdbc]))
  (:import  (java.io      BufferedInputStream)
            (java.util    Locale MissingResourceException Properties ResourceBundle ResourceBundle$Control TimeZone)))

;; Localization

(def ^:dynamic *locale*
  (Locale/getDefault))

(def ^:private resource-bundle-control
  (proxy [ResourceBundle$Control] []
    (newBundle [base-name locale format class-loader reload?]
      (let [resource-name (.toResourceName this (.toBundleName this base-name locale) "properties.xml")]
        (with-open [input-stream (-> (if reload?
                                       (-> (doto (.openConnection (.getResource class-loader resource-name))
                                             (.setUseCaches false))
                                           (.getInputStream))
                                       (.getResourceAsStream class-loader resource-name))
                                     (BufferedInputStream.))]
          (let [properties (doto (Properties.)
                             (.loadFromXML input-stream))]
            (proxy [ResourceBundle] []
              (handleGetObject [key]
                (.getProperty properties key)))))))
    (getFallbackLocale [base-name locale]
      (Locale. ""))))

(defn load-messages
  []
  (ResourceBundle/getBundle "messages"
                            *locale*
                            resource-bundle-control))

(def ^:dynamic *messages*
  (let [locale (Locale/getDefault)]
    (load-messages)))

(defn localized-string
  [key]
  (try
    (let [string (.getString *messages* key)]
      (cond->> string
        (.contains string "\n") (.trim)))
    (catch MissingResourceException _
      (name key))))

(TimeZone/setDefault (TimeZone/getTimeZone "GMT"))  ; JDBC will add default timezone to PreparedStatement parameters...

;; Aspect Oriented Programming

(defn weave-aspect
  [namespace function-name-re aspect]
  (doseq [[symbol var] (->> (ns-interns namespace)
                            (filter (fn [[symbol var]]
                                      (re-matches function-name-re (name symbol)))))]
    (let [function @var]
      (.bindRoot var (fn [& args] (aspect symbol function args))))))

;; Database

(def as-quoted-sql-name
   (jdbc/as-sql-name (jdbc/quoted \")))

(defn db-metadata [db function]
  (with-open [connection (jdbc/get-connection db)]
    (->> (-> connection
             (.getMetaData)
             (function))
         (jdbc/result-set-seq)
         (into []))))

;; Misc

(defmacro condp'
  [expr & forms]
  `(condp ~(fn [f x] (f x)) ~expr ~@forms))

(def not-blank?
  (complement string/blank?))

(def not-empty?
  (complement empty?))

(defn doall-recur
  [x]
  (condp' x
    seq?    (doall (map doall-recur x))
    list?   (list  (map doall-recur x))
    vector? (vec   (map doall-recur x))
    x))

(defn capitalize'
  [x]
  (str (string/upper-case (.substring x 0 1))
       (.substring x 1)))

(defn array-map'
  [keyvals]
  (reduce (fn [result [key val]] (assoc result key val)) {} keyvals))
