(ns exchange.db.datomic
  (:import (datomic.db Db)
           (java.util Map))
  (:require [earthworm.util.datomic :refer [entities attr lookup-eid init-conn]]
            [datomic.api :as d]
            [schema.core :as s]
            [exchange.db.protocol :as dp]
            [exchange.util :as u]))

; ==============================================
; 数据库定义
; ==============================================

(def user
  [(attr :user/name :db.type/string "用户名" :db/unique :db.unique/identity)
   (attr :user/ticket :db.type/ref "奖券" :db/cardinality :db.cardinality/many :db/isComponent true)])

(def ticket
  [(attr :ticket/topic :db.type/string "活动主题")
   (attr :ticket/name :db.type/string "奖券名称")
   (attr :ticket/num :db.type/long "奖券数量")
   (attr :ticket/exchanged :db.type/long "已兑换数量")])

(def record
  [(attr :record/order-id :db.type/string "订单号" :db/unique :db.unique/identity)
   (attr :record/topic :db.type/string "活动主题")
   (attr :record/time :db.type/long "兑奖时间")
   (attr :record/username :db.type/string "用户名")
   (attr :record/ticket :db.type/string "消耗的奖券名")
   (attr :record/cost :db.type/long "消耗的奖券数量")
   (attr :record/prize :db.type/string "奖品名")
   (attr :record/type :db.type/keyword "奖品类型")
   (attr :record/provided? :db.type/boolean "奖品信息是否提供的标志")
   (attr :record/card :db.type/string "卡密奖品兑奖卡账号")
   (attr :record/code :db.type/string "卡密奖品兑奖卡密码")
   (attr :record/address :db.type/string "实物奖品兑奖地址")
   (attr :record/postcode :db.type/string "实物奖品兑奖邮编")
   (attr :record/recipient :db.type/string "实物奖品兑奖收件人")
   (attr :record/phone :db.type/string "实物奖品兑奖手机")])

(def db-schema (entities :db.part/db (concat user ticket record)))

; ==============================================
; 工具函数
; ==============================================

(defn add-key-ns
  "给keyword添加命名空间"
  [ns k]
  (keyword (str ns "/" (name k))))

(defn rm-key-ns
  "去除keyword的命名空间"
  [k]
  (keyword (name k)))

(s/defn get-userid :- (s/maybe Long)
  "获取用户的数据库id"
  [db :- Db username :- s/Str]
  (lookup-eid db [[:user/name username]]))

(s/defn get-ticketid :- (s/maybe Long)
  "获取用户下具体活动的奖券信息"
  [db :- Db userid :- Long topic :- s/Str ticket :- s/Str]
  (lookup-eid db [userid [:user/ticket {:ticket/topic topic :ticket/name ticket}]]))

(s/defn output-format :- Map
  "将数据库中拉取数据"
  [db-data :- Map]
  (-> db-data
      (dissoc :db/id)
      ((partial u/map-keys rm-key-ns))))

(s/defn pull-data
  "根据ID,从数据库中拉取数据"
  [db :- Db id :- (s/maybe Long)]
  (when id (output-format (d/pull db '[*] id))))

(defn do-fn-after-trans
  "执行数据库语句后,再执行传入的数据库函数"
  [db-conn tx-data db-fn]
  (let [tx-result @(d/transact db-conn tx-data)]
    (db-fn (:db-after tx-result))))

; ==============================================
; 数据库语句生成函数
; ==============================================

(s/defn new-ticket-tx-data
  "用户下没有该主题的奖券信息,新建一个主题的奖券"
  [topic :- s/Str ticket :- s/Str num :- Long]
  [{:db/id            (d/tempid :db.part/user)
    :ticket/topic     topic
    :ticket/name      ticket
    :ticket/num       num
    :ticket/exchanged 0}])

(s/def TicketData
  "奖券信息结构"
  {:username              s/Str
   :topic                 s/Str
   :name                  s/Str
   (s/optional-key :time) Long
   (s/optional-key :game) s/Str
   (s/optional-key :win)  Long
   })

(s/defn add-ticket-tx-data
  "增加用户的奖券"
  [db :- Db {:keys [username topic name] :as ticket} :- (s/maybe TicketData)]
  (when ticket
    (if-let [uid (get-userid db username)]
      (if (get-ticketid db uid topic name)
        [[:db.fn/smart-atom-plus
          [uid [:user/ticket {:ticket/topic topic :ticket/name name}]]
          :ticket/num 1]]
        [{:db/id       uid
          :user/ticket (new-ticket-tx-data topic name 1)}])
      [{:db/id       (d/tempid :db.part/user)
        :user/name   username
        :user/ticket (new-ticket-tx-data topic name 1)}])))

(s/def TicketDelta
  "修改用户奖券信息的结构"
  {:name s/Str
   :num  Long})

(s/defn update-ticket-tx-data
  "获取更新用户奖券的数据库语句"
  [db :- Db username :- s/Str topic :- s/Str {:keys [name num]} :- TicketDelta]
  (when-let [uid (get-userid db username)]
    (if (get-ticketid db uid topic name)
      [[:db.fn/smart-atom-plus [uid [:user/ticket {:ticket/topic topic :ticket/name name}]] :ticket/num num]]
      [{:db/id       uid
        :user/ticket (new-ticket-tx-data topic name num)}])))

(s/defn update-tickets-tx-data
  "获取更新用户奖券的数据库语句"
  [db :- Db username :- s/Str topic :- s/Str deltas :- [TicketDelta]]
  (->> (map (partial update-ticket-tx-data db username topic) deltas)
       (apply concat)))

(s/def ExchangeRecord
  "兑换记录结构"
  {:order-id                   s/Str
   :topic                      s/Str
   :time                       Long
   :username                   s/Str
   :ticket                     s/Str
   :cost                       Long
   :prize                      s/Str
   :type                       s/Keyword
   :provided?                  s/Bool
   (s/optional-key :card)      s/Str
   (s/optional-key :code)      s/Str
   (s/optional-key :add-money) Long
   (s/optional-key :finance)   Long
   (s/optional-key :address)   s/Str
   (s/optional-key :postcode)  s/Str
   (s/optional-key :recipient) s/Str
   (s/optional-key :phone)     s/Str})

(s/defn filter-record
  [record :- ExchangeRecord]
  (select-keys record [:order-id :topic :time :username :ticket :cost
                       :prize :type :provided? :card :code
                       :address :postcode :recipient :phone]))

(s/defn exchange-ticket-tx-data
  "获取数据库语句,添加一条兑换记录,并扣除奖券"
  [db :- Db {:keys [username topic ticket cost] :as record} :- ExchangeRecord]
  (let [uid (get-userid db username)
        tid (get-ticketid db uid topic ticket)]
    (when (and uid tid)
      [(-> (filter-record record)
           ((partial u/map-keys (partial add-key-ns "record")))
           (assoc :db/id (d/tempid :db.part/user)))
       [:db.fn/smart-atom-plus
        [uid [:user/ticket {:ticket/topic topic :ticket/name ticket}]]
        :ticket/exchanged cost]
       [:db.fn/smart-atom-plus
        [uid [:user/ticket {:ticket/topic topic :ticket/name ticket}]]
        :ticket/num (- cost)]])))

(s/defn update-record-tx-data
  "获取数据库语句,更新一条兑换记录"
  [db :- Db {:keys [order-id] :as record} :- ExchangeRecord]
  (when-let [rid (lookup-eid db [[:record/order-id order-id]])]
    [(-> (filter-record record)
         ((partial u/map-keys (partial add-key-ns "record")))
         (assoc :db/id rid))]))

; ==============================================
; 数据库接口定义
; ==============================================

(s/defn get-tickets
  "获取用户的奖券信息"
  [db :- Db topic :- s/Str username :- s/Str]
  (->> (d/q '[:find [(pull ?tid [*]) ...]
              :in $ ?topic ?username
              :where
              [?uid :user/name ?username]
              [?uid :user/ticket ?tid]
              [?tid :ticket/topic ?topic]]
            db topic username)
       (map output-format)))

(s/defn get-ticket
  "获取用户的奖券信息"
  [db :- Db topic :- s/Str username :- s/Str ticket :- s/Str]
  (when-let [uid (get-userid db username)]
    (pull-data db (get-ticketid db uid topic ticket))))

(defrecord DatomicExchangeDb [db-conn]
  dp/ExchangeDb
  (user-tickets [_ topic username]
    (get-tickets (d/db db-conn) topic username))

  (add-ticket [_ {:keys [username topic name] :as ticket}]
    (when-let [tx-data (add-ticket-tx-data (d/db db-conn) ticket)]
      (do-fn-after-trans db-conn tx-data #(get-ticket % topic username name))))

  (update-tickets [_ topic username tickets-delta]
    (when-let [tx-data (update-tickets-tx-data (d/db db-conn) username topic tickets-delta)]
      (do-fn-after-trans db-conn tx-data #(get-tickets % topic username))))

  (exchange-ticket [_ {:keys [username topic ticket] :as record}]
    (when-let [tx-data (exchange-ticket-tx-data (d/db db-conn) record)]
      (do-fn-after-trans db-conn tx-data #(get-ticket % topic username ticket))))

  (update-record [_ {:keys [order-id] :as record}]
    (when-let [tx-data (update-record-tx-data (d/db db-conn) record)]
      @(d/transact db-conn tx-data)))

  (user-record [_ order-id]
    (let [db (d/db db-conn)]
      (pull-data db (lookup-eid db [[:record/order-id order-id]]))))

  (user-records [_ topic username]
    (->> (d/q '[:find [(pull ?e [*]) ...]
                :in $ ?topic ?username
                :where [?e :record/topic ?topic] [?e :record/username ?username]]
              (d/db db-conn) topic username)
         (map output-format)))

  (prize-records [_ topic prize-name]
    (->> (d/q '[:find [(pull ?e [*]) ...]
                :in $ ?topic ?prize
                :where [?e :record/topic ?topic] [?e :record/prize ?prize]]
              (d/db db-conn) topic prize-name)
         (map output-format)))

  (prizes-count [_ topic [start end] prizes]
    (d/q '[:find ?prize (count ?e)
           :in $ ?topic ?start ?end [?prize ...]
           :where
           [?e :record/topic ?topic]
           [?e :record/prize ?prize]
           [?e :record/time ?time]
           [(<= ?start ?time)]
           [(< ?time ?end)]]
         (d/db db-conn) topic start end prizes))

  (recent-records [_ topic prize-filter num]
    (->> (d/q '[:find [(pull ?e [:db/id :record/time]) ...]
                :in $ ?topic ?prize-filter
                :where
                [?e :record/topic ?topic]
                [?e :record/type ?type]
                [(contains? ?prize-filter ?type)]]
              (d/db db-conn) topic prize-filter)
         (sort-by (comp - :record/time))
         (take num)
         (map #(pull-data (d/db db-conn) (:db/id %)))))

  (users-exchanged-ticket [_ topic ticket-name]
    (d/q '[:find ?username ?exchanged
           :in $ ?topic ?ticket
           :where
           [?uid :user/name ?username]
           [?uid :user/ticket ?tid]
           [?tid :ticket/topic ?topic]
           [?tid :ticket/name ?ticket]
           [?tid :ticket/exchanged ?exchanged]
           [(pos? ?exchanged)]]
         (d/db db-conn) topic ticket-name))
  )

(defmethod dp/mk-exchange-db :datomic
  [{:keys [url]}]
  (map->DatomicExchangeDb {:db-conn (init-conn {:schema db-schema} url false)}))

(comment
  (defn mk-save-express
    "保存快递单号服务构造器"
    [db]
    (fn [{:keys [order-id] :as inf}]
      (let [record (dp/user-record db order-id)]
        (log/debug {:svr_method "save-express" :record record :inf inf})
        (when record
          (let [new-record (-> (select-keys inf [:express-id])
                               (merge record))]
            (dp/update-record db new-record))))))
  (defn 保存实物奖品发货信息
    "被外部使用,保存地址等信息到指定的兑换记录"
    [保存服务 {:keys [order-id express- express-id] :as inf}]
    ((:fn-save-express 保存服务) inf))
  :fn-save-express (mk-save-express db))